#include <solomon/services/fetcher/lib/config_updater/config_updater.h>

#include <solomon/services/fetcher/lib/cluster/cluster.h>
#include <solomon/services/fetcher/lib/data_sink/processing_client.h>

#include <solomon/services/fetcher/testlib/actor_system.h>
#include <solomon/services/fetcher/testlib/coremon.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/stream/mem.h>

using namespace testing;
using namespace NActors;
using namespace NMonitoring;
using namespace NSolomon::NFetcher;
using namespace NSolomon::NTesting;
using namespace NSolomon;
using namespace NThreading;

namespace NSolomon::NFetcher {

bool operator==(const TShardAssignment& lhs, const TShardAssignment& rhs) {
    return lhs.Location == rhs.Location && lhs.Shards == rhs.Shards;
}

} // namespace NSolomon::NFetcher

static const TString CLUSTER = TStringBuilder()
    << "node0:0\n"
    << "node1:0\n"
    << FQDNHostName() << ":0\n"
    << "node3:0\n"
    << "node4:0\n"
    << "node5:0\n"
    << "node6:0\n"
    << "node7:0\n"
    << "node8:0\n"
    << "node9:0";

class TShardMapBuilderTest: public ::testing::Test {
public:
    void SetUp() override {
        ActorRuntime_.Reset(new NSolomon::NTesting::TFetcherActorRuntime);
        ActorRuntime_->Initialize();

        EdgeId_ = ActorRuntime_->AllocateEdgeActor();

        Cluster_ = MakeCluster(CLUSTER);
        Coremons_ = std::make_shared<TMockCoremonCluster>(Cluster_);

        ShardMapBuilder_ = ActorRuntime_->Register(CreateShardMapBuilderRequestingLeader(WrapClusterClient(Coremons_), Cluster_));
    }

    void TearDown() override {
        ActorRuntime_->Send(ShardMapBuilder_, EdgeId_, MakeHolder<TEvents::TEvPoison>());
        ActorRuntime_->GrabEdgeEvent<TEvents::TEvActorDied>(EdgeId_);

        ActorRuntime_.Reset();
    }

protected:
    TMockCoremonClient* GetClient(const TClusterNode& loc) {
        return static_cast<TMockCoremonClient*>(Coremons_->Get(loc.Endpoint));
    }

    TMockCoremonClient* GetLocalClient() {
        return GetClient(Cluster_->Local());
    }

    static IClusterMapPtr MakeCluster(TStringBuf data) {
        TMemoryInput is{data};
        TMockMapper m;
        return NSolomon::TClusterMapBase::Load(is, &m);
    }

protected:
    THolder<NSolomon::NTesting::TFetcherActorRuntime> ActorRuntime_;
    TActorId ShardMapBuilder_;
    IClusterMapPtr Cluster_;
    NCoremon::ICoremonClusterClientPtr Coremons_;
    TActorId EdgeId_;
};

TEST_F(TShardMapBuilderTest, LocationRemapping) {
    auto* client = GetLocalClient();
    auto leaderLocation = Cluster_->NodeById(0);

    TVector<NCoremon::TShardAssignments::THost> hosts;
    auto locs = Cluster_->Nodes();
    for (const auto& loc: locs) {
        hosts.emplace_back(loc.Fqdn, static_cast<ui32>(loc.NodeId + 1) % locs.size());
    }

    client->SetShardAssignments(NCoremon::TShardAssignments {
            .Leader = leaderLocation->Fqdn,
            .Assignments = {
                    {hosts[0], {0, 1, 2}},
                    {hosts[1], {3, 4, 5}},
                    {hosts[2], {6, 7, 8}},
            },
    });

    auto sender = ActorRuntime_->AllocateEdgeActor(0);
    ActorRuntime_->Send(new IEventHandle(ShardMapBuilder_, sender, new TEvLoadShardMap{}));
    auto ev = ActorRuntime_->GrabEdgeEvent<TEvLoadShardMapResponse>(sender);
    ASSERT_TRUE(ev->Get()->Result.Success());
    auto val = ev->Get()->Result.Extract();
    auto loc = locs.begin();
    ASSERT_THAT(val, UnorderedElementsAre(
            TShardAssignment{*loc, {0, 1, 2}},
            TShardAssignment{*(++loc), {3, 4, 5}},
            TShardAssignment{*(++loc), {6, 7, 8}}
    ));
}

TEST_F(TShardMapBuilderTest, ShardMapIsLoadedOnDemand) {
    auto* client = GetLocalClient();
    auto leaderLocation = Cluster_->NodeById(0);

    TVector<NCoremon::TShardAssignments::THost> hosts;
    auto locs = Cluster_->Nodes();
    for (const auto& loc: locs) {
        hosts.emplace_back(loc.Fqdn, static_cast<ui32>(loc.NodeId));
    }

    client->SetShardAssignments(NCoremon::TShardAssignments {
            .Leader = leaderLocation->Fqdn,
            .Assignments = {
                    {hosts[0], {0, 1, 2}},
                    {hosts[1], {3, 4, 5}},
                    {hosts[2], {6, 7, 8}},
            },
    });

    auto sender = ActorRuntime_->AllocateEdgeActor(0);
    ActorRuntime_->Send(new IEventHandle(ShardMapBuilder_, sender, new TEvLoadShardMap{}));
    auto ev = ActorRuntime_->GrabEdgeEvent<TEvLoadShardMapResponse>(sender);
    ASSERT_TRUE(ev->Get()->Result.Success());
    auto val = ev->Get()->Result.Extract();
    auto loc = locs.begin();
    ASSERT_THAT(val, UnorderedElementsAre(
            TShardAssignment{*loc, {0, 1, 2}},
            TShardAssignment{*(++loc), {3, 4, 5}},
            TShardAssignment{*(++loc), {6, 7, 8}}
    ));
}

TEST_F(TShardMapBuilderTest, LeaderIsChangedOnResponse) {
    auto leaderLocation = Cluster_->NodeById(0);
    ASSERT_THAT(leaderLocation, Not(Eq(Cluster_->Local())));

    TMockCoremonClient* client;
    for (const auto& loc: Cluster_->Nodes()) {
        client = GetClient(loc);
        client->SetShardAssignments(NCoremon::TShardAssignments {
                .Leader = leaderLocation->Fqdn,
                .Assignments = { },
        });
    }

    client = GetLocalClient();

    ASSERT_THAT(client->AssignmentCalls, Eq(0));
    auto sender = ActorRuntime_->AllocateEdgeActor(0);
    ActorRuntime_->Send(new IEventHandle(ShardMapBuilder_, sender, new TEvLoadShardMap{}));
    auto ev = ActorRuntime_->GrabEdgeEvent<TEvLoadShardMapResponse>(sender);
    ASSERT_THAT(client->AssignmentCalls, Eq(1));

    // all requests must go to the leader now
    client = GetClient(*leaderLocation);
    ASSERT_THAT(client->AssignmentCalls, Eq(0));
    ActorRuntime_->Send(new IEventHandle(ShardMapBuilder_, sender, new TEvLoadShardMap{}));
    ev = ActorRuntime_->GrabEdgeEvent<TEvLoadShardMapResponse>(sender);
    ASSERT_THAT(client->AssignmentCalls, Eq(1));
}

TEST_F(TShardMapBuilderTest, LeaderFallbackIsUsedOnUnknownLocation) {
    // fixed random seed, to make leader switch more deterministic
    SetRandomSeed(57);

    for (const auto& node: Cluster_->Nodes()) {
        TMockCoremonClient* client = GetClient(node);
        client->SetShardAssignments(NCoremon::TShardAssignments {
                .Leader = "unknown_node",
                .Assignments = { },
        });
    }

    auto* localClient = GetLocalClient();
    auto sender = ActorRuntime_->AllocateEdgeActor(0);

    // first call will go to a local host
    {
        ActorRuntime_->Send(new IEventHandle(ShardMapBuilder_, sender, new TEvLoadShardMap{}));
        ActorRuntime_->GrabEdgeEvent<TEvLoadShardMapResponse>(sender);
        ASSERT_EQ(localClient->AssignmentCalls, 1);
    }

    // next call will go to a random client
    {
        ActorRuntime_->Send(new IEventHandle(ShardMapBuilder_, sender, new TEvLoadShardMap{}));
        ActorRuntime_->GrabEdgeEvent<TEvLoadShardMapResponse>(sender);
        ASSERT_EQ(localClient->AssignmentCalls, 1);
    }

    // and next time we will try to call local client again
    {
        ActorRuntime_->Send(new IEventHandle(ShardMapBuilder_, sender, new TEvLoadShardMap{}));
        ActorRuntime_->GrabEdgeEvent<TEvLoadShardMapResponse>(sender);
        ASSERT_EQ(localClient->AssignmentCalls, 2);
    }
}

TEST_F(TShardMapBuilderTest, CacheWorks) {
    auto* client = GetLocalClient();
    const auto loc = Cluster_->Local();
    client->SetShardAssignments(NCoremon::TShardAssignments{
            .Leader = loc.Fqdn,
            .Assignments = {
                    {{loc.Fqdn, loc.NodeId}, {0, 1, 2}},
            },
    });

    auto sender = ActorRuntime_->AllocateEdgeActor(0);
    ActorRuntime_->Send(new IEventHandle(ShardMapBuilder_, sender, new TEvLoadShardMap{}));
    auto ev = ActorRuntime_->GrabEdgeEvent<TEvLoadShardMapResponse>(sender);
    ASSERT_THAT(client->AssignmentCalls, Eq(1));
    ASSERT_TRUE(ev->Get()->Result.Success());
    auto initial = ev->Get()->Result.Extract();

    auto loadEv = MakeHolder<TEvLoadShardMap>();
    loadEv->CachePolicy = TDuration::Hours(5);
    ActorRuntime_->Send(new IEventHandle(ShardMapBuilder_, sender, loadEv.Release()));

    ev = ActorRuntime_->GrabEdgeEvent<TEvLoadShardMapResponse>(sender);
    ASSERT_TRUE(ev->Get()->Result.Success());
    auto cached = ev->Get()->Result.Extract();
    ASSERT_THAT(client->AssignmentCalls, Eq(1));
    ASSERT_THAT(cached, Eq(initial));
}
