#include <solomon/services/fetcher/lib/config_updater/config_updater.h>

#include <solomon/libs/cpp/conf_db/table_loader/table_loader.h>
#include <solomon/libs/cpp/conf_db/db.h>
#include <solomon/services/fetcher/testlib/actor_system.h>
#include <solomon/services/fetcher/testlib/db.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/actorsystem.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/actors/core/executor_pool_basic.h>
#include <library/cpp/actors/core/scheduler_basic.h>

#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/string/builder.h>
#include <util/system/event.h>
#include <util/system/hostname.h>

using namespace testing;
using namespace NActors;
using namespace NSolomon::NFetcher;
using namespace NSolomon::NTesting;
using namespace NSolomon::NDb;
using namespace NSolomon::NTableLoader;

namespace {
    struct TMockShardMapBuilder: public TActorBootstrapped<TMockShardMapBuilder> {
        explicit TMockShardMapBuilder(const TVector<ui32>& shardIds)
            : ShardIds_(shardIds)
        {
        }

        void Bootstrap(const TActorContext&) {
            BootstrapImpl();
        }

        virtual void BootstrapImpl() {
            Become(&TThis::StateFunc);
        }

        STFUNC(StateFunc) {
            switch (ev->GetTypeRewrite()) {
                HFunc(TEvLoadShardMap, OnLoadShardMap);
            }
        }

    private:
        void OnLoadShardMap(const TEvLoadShardMap::TPtr& ev, const TActorContext&) {
            auto value = TLoadShardMapResult::FromValue(
                TVector<TShardAssignment>{TShardAssignment{{FQDNHostName(), 0, 0}, ShardIds_}});
            Send(ev->Sender, new TEvLoadShardMapResponse{std::move(value)});
        }

        const TVector<ui32>& ShardIds_;
    };

    struct TFailingInitBuilder: TMockShardMapBuilder {
        TFailingInitBuilder(ui16 failCount, const TVector<ui32>& shardIds, bool respondWithError)
            : TMockShardMapBuilder{shardIds}
            , SendError_{respondWithError}
            , FailCount_{failCount}
        {
        }

        void BootstrapImpl() override {
            Become(&TFailingInitBuilder::StateInit);
        }

        STFUNC(StateInit) {
            Y_UNUSED(ctx);
            switch (ev->GetTypeRewrite()) {
                hFunc(TEvLoadShardMap, Handle);
            }
        }

    private:
        void Handle(const TEvLoadShardMap::TPtr& ev) {
            if (FailCount_ > 0) {
                if (SendError_) {
                    Send(ev->Sender, new TEvLoadShardMapResponse{TLoadShardMapResult::FromError("error")});
                } else {
                    Send(ev->Sender, new TEvLoadShardMapResponse{TLoadShardMapResult::FromValue(TVector<TShardAssignment>{})});
                }
            }

            --FailCount_;
            if (FailCount_ == 0) {
                Become(&TMockShardMapBuilder::StateFunc);
            }
        }

    private:
        bool SendError_;
        i32 FailCount_;
    };
} // namespace

// TODO: add update tests when there are corresponding methods in dao
class TShardUpdaterTest: public ::testing::Test {
public:
    void SetUp() override {
        ActorSystem_ = MakeActorSystem(true);
        ActorSystem_->AddLocalService(MakeShardMapBuilderId(),
            TActorSetupCmd{new TMockShardMapBuilder(LocalShardIds_), TMailboxType::Simple, 0}
        );

        ActorSystem_->Initialize();

        Db_.Reset();
        FillDatabase();

        auto serviceDao = Db_.Services();
        auto shardDao = Db_.Shards();
        auto clusterDao = Db_.Clusters();
        auto agentDao = Db_.Agents();
        auto providerDao = Db_.Providers();

        auto* shardUpdater = CreateConfigUpdaterActor(TConfigUpdaterConfig{
            .ServiceDao = serviceDao,
            .ShardDao = shardDao,
            .ClusterDao = clusterDao,
            .AgentDao = agentDao,
            .ProviderDao = providerDao,
            .Interval = TDuration::Seconds(5),
            .MetricRegistry = MetricRegistry_,
            .Mode = EUpdaterMode::All,
        });
        UpdaterId_ = ActorSystem_->Register(shardUpdater);
        ActorSystem_->WaitForBootstrap();

        TestActorId_ = ActorSystem_->AllocateEdgeActor();

        auto waitActor = ActorSystem_->AllocateEdgeActor();
        ActorSystem_->Send(new IEventHandle{UpdaterId_, waitActor, new TEvents::TEvSubscribe});

        bool dataLoaded = false;
        while (!dataLoaded) {
            auto ev = ActorSystem_->GrabEdgeEvent<TEvConfigChanged>(waitActor);
            dataLoaded = ev->Get()->Added.size() > 0;
        }

        ActorSystem_->Send(new IEventHandle{UpdaterId_, TestActorId_, new TEvents::TEvSubscribe});
    }

    void TearDown() override {
        ActorSystem_.Reset();
        DbShards_.clear();
        DbClusters_.clear();
        DbServices_.clear();
        LocalShardIds_.clear();
    }

protected:
    void FillDatabase() {
        for (auto i = 0; i < 2; ++i) {
            NewService();
        }

        for (auto i = 0; i < 2; ++i) {
            auto cluster = MakeClusterConfig();
            cluster.Id = TStringBuilder() << "cluster_id_" << i;
            cluster.Name = TStringBuilder() << "cluster_name_" << i;
            DbClusters_.push_back(cluster);
        }

        for (auto i = 0; i < 4; ++i) {
            const auto serviceIdx = i & 1;
            const auto clusterIdx = (i & 2) >> 1;
            auto& cluster = DbClusters_[clusterIdx];
            auto& service = DbServices_[serviceIdx];

            NewShard(service, cluster, true);
        }

        Db_.InitTables(DbServices_, DbClusters_, DbShards_);
    }

    NModel::TServiceConfig& NewService() {
        NModel::TServiceConfig service = MakeServiceConfig();
        service.Id = TStringBuilder() << "service_id_" << DbServices_.size();
        service.Name = TStringBuilder() << "service_name_" << DbServices_.size();
        service.ShardSettings.PullOrPush = NModel::TPullSettings{.Path = "/something"};
        DbServices_.push_back(service);

        return DbServices_.back();
    }

    NModel::TShardConfig& NewShard(const NModel::TServiceConfig& service, const NModel::TClusterConfig& cluster, bool local) {
        NModel::TShardConfig shard = MakeShardConfig();
        shard.NumId = DbShards_.size(); // NumId must be globally unique
        shard.Id = TStringBuilder() << "shard_id_" << DbShards_.size();

        if (local) {
            LocalShardIds_.push_back(shard.NumId);
        }

        shard.ServiceId = service.Id;
        shard.ServiceName = service.Name;

        shard.ClusterName = cluster.Name;
        shard.ClusterId = cluster.Id;

        DbShards_.push_back(shard);
        return DbShards_.back();
    }

protected:
    TVector<NModel::TServiceConfig> DbServices_;
    TVector<NModel::TClusterConfig> DbClusters_;
    TVector<NModel::TShardConfig> DbShards_;
    TVector<ui32> LocalShardIds_;

    TActorId UpdaterId_;
    TActorId TestActorId_;
    THolder<TFetcherActorRuntime> ActorSystem_;
    TTestDb Db_;
    NMonitoring::TMetricRegistry MetricRegistry_;
};

TEST_F(TShardUpdaterTest, SubscriberGetsAllShardsOnSubscribe) {
    TVector<TShardId> expectedShards;
    Transform(
            DbShards_.begin(),
            DbShards_.end(),
            std::back_inserter(expectedShards),
            [] (auto shard) {
                return TShardId{shard.Id, shard.NumId};
            }
    );

    TVector<TShardId> shards;
    auto ev = ActorSystem_->GrabEdgeEvent<TEvConfigChanged>(TestActorId_);

    for (auto&& shard: ev->Get()->Added) {
        shards.push_back(shard.Id());
    }

    ASSERT_THAT(shards, UnorderedElementsAreArray(expectedShards));
}

TEST_F(TShardUpdaterTest, SubscriberGetsNotifiedOnDelete) {
    auto ev = ActorSystem_->GrabEdgeEvent<TEvConfigChanged>(TestActorId_);

    const auto projectId = MakeShardConfig().ProjectId;
    Db_.Shards()->Delete("shard_id_1", projectId).GetValueSync();
    ActorSystem_->Send(UpdaterId_, MakeHolder<TEvents::TEvWakeup>());

    TVector<TInfoOfAShardToRemove> removed;
    do {
        ev = ActorSystem_->GrabEdgeEvent<TEvConfigChanged>(TestActorId_);
        removed = ev->Get()->Removed;
    } while (removed.empty());

    ASSERT_THAT(
            removed,
            UnorderedElementsAre(TInfoOfAShardToRemove{ TShardId{"shard_id_1", 1}, EFetcherShardType::Simple }));
}

TEST_F(TShardUpdaterTest, SubscriberGetsNotifiedOnAddition) {
    auto ev = ActorSystem_->GrabEdgeEvent<TEvConfigChanged>(TestActorId_);

    auto& service = NewService();
    auto& shard = NewShard(service, DbClusters_.back(), true);
    Db_.Services()->Insert(service).GetValueSync();
    Db_.Shards()->Insert(shard).GetValueSync();

    ActorSystem_->Send(UpdaterId_, MakeHolder<TEvents::TEvWakeup>());

    ev = ActorSystem_->GrabEdgeEvent<TEvConfigChanged>(TestActorId_);
    TVector<TShardId> added;
    do {
        ev = ActorSystem_->GrabEdgeEvent<TEvConfigChanged>(TestActorId_);
        for (auto&& shard: ev->Get()->Added) {
            added.push_back(shard.Id());
        }
    } while (added.empty());


    ASSERT_THAT(added, UnorderedElementsAre(TShardId{"shard_id_4", 4}));
}

TEST_F(TShardUpdaterTest, ServiceDeletion) {
    auto ev = ActorSystem_->GrabEdgeEvent<TEvConfigChanged>(TestActorId_);

    const auto projectId = MakeShardConfig().ProjectId;
    Db_.Services()->Delete("service_id_1", projectId).GetValueSync();
    ActorSystem_->Send(UpdaterId_, MakeHolder<TEvents::TEvWakeup>());

    TVector<TInfoOfAShardToRemove> removed;
    do {
        ev = ActorSystem_->GrabEdgeEvent<TEvConfigChanged>(TestActorId_);
        removed = ev->Get()->Removed;
    } while (removed.empty());

    ASSERT_THAT(
            removed,
            UnorderedElementsAre(
                    TInfoOfAShardToRemove{ TShardId{"shard_id_1", 1}, EFetcherShardType::Simple },
                    TInfoOfAShardToRemove{ TShardId{"shard_id_3", 3}, EFetcherShardType::Simple }));
}

TEST_F(TShardUpdaterTest, ClusterDeletion) {
    auto ev = ActorSystem_->GrabEdgeEvent<TEvConfigChanged>(TestActorId_);

    const auto projectId = MakeShardConfig().ProjectId;
    Db_.Clusters()->Delete("cluster_id_1", projectId).GetValueSync();
    ActorSystem_->Send(UpdaterId_, MakeHolder<TEvents::TEvWakeup>());

    TVector<TInfoOfAShardToRemove> removed;
    do {
        ev = ActorSystem_->GrabEdgeEvent<TEvConfigChanged>(TestActorId_);
        removed = ev->Get()->Removed;
    } while (removed.empty());

    ASSERT_THAT(
            removed,
            UnorderedElementsAre(
                    TInfoOfAShardToRemove{ TShardId{"shard_id_2", 2}, EFetcherShardType::Simple },
                    TInfoOfAShardToRemove{ TShardId{"shard_id_3", 3}, EFetcherShardType::Simple }));
}

TEST_F(TShardUpdaterTest, AgentAddition) {
    Y_UNUSED(ActorSystem_->GrabEdgeEvent<TEvConfigChanged>(TestActorId_));
    auto agent1 = MakeAgentConfig();
    auto agent2 = MakeAgentConfig();
    auto provider1 = MakeProviderConfig();
    auto provider2 = MakeProviderConfig();

    agent1.Provider = provider1.Id = "provider1";
    agent2.Provider = provider2.Id = "provider2";

    Db_.Agents()->InsertOrUpdate(agent1).GetValueSync();
    Db_.Agents()->InsertOrUpdate(agent2).GetValueSync();
    Db_.Providers()->Insert(provider1).GetValueSync();
    Db_.Providers()->Insert(provider2).GetValueSync();

    ActorSystem_->Send(UpdaterId_, MakeHolder<TEvents::TEvWakeup>());
    TEvConfigChanged::TPtr ev;
    do {
        ev = ActorSystem_->GrabEdgeEvent<TEvConfigChanged>(TestActorId_);
    } while (ev->Get()->Added.empty());

    ASSERT_THAT(ev->Get()->Removed, IsEmpty());
    auto& added = ev->Get()->Added;
    ASSERT_THAT(added, SizeIs(2));

    TVector<TStringBuf> expectedProviderIds{ provider1.Id, provider2.Id };
    TVector<TStringBuf> actualIds{ added[0].ProjectId(), added[1].ProjectId() };
    ASSERT_THAT(actualIds, UnorderedElementsAreArray(expectedProviderIds));

    ASSERT_THAT(added[0].Type(), EFetcherShardType::Agent);
    ASSERT_THAT(added[1].Type(), EFetcherShardType::Agent);
}

TEST_F(TShardUpdaterTest, AgentAdditionWithoutProvider) {
    Y_UNUSED(ActorSystem_->GrabEdgeEvent<TEvConfigChanged>(TestActorId_));
    auto agent1 = MakeAgentConfig();
    auto agent2 = MakeAgentConfig();
    agent1.Provider = "provider1";
    agent2.Provider = "provider2";

    auto provider1 = MakeProviderConfig();
    provider1.Id = "provider1";

    Db_.Agents()->InsertOrUpdate(agent1).GetValueSync();
    Db_.Agents()->InsertOrUpdate(agent2).GetValueSync();
    Db_.Providers()->Insert(provider1).GetValueSync();

    ActorSystem_->Send(UpdaterId_, MakeHolder<TEvents::TEvWakeup>());
    TEvConfigChanged::TPtr ev;
    do {
        ev = ActorSystem_->GrabEdgeEvent<TEvConfigChanged>(TestActorId_);
    } while (ev->Get()->Added.empty());

    ASSERT_THAT(ev->Get()->Removed, IsEmpty());
    auto& added = ev->Get()->Added;
    ASSERT_THAT(added, SizeIs(1));
    ASSERT_THAT(added[0].ProjectId(), agent1.Provider);
    ASSERT_THAT(added[0].Type(), EFetcherShardType::Agent);
}

TEST_F(TShardUpdaterTest, AgentAdditionToExistingShard) {
    Y_UNUSED(ActorSystem_->GrabEdgeEvent<TEvConfigChanged>(TestActorId_));
    auto agent = MakeAgentConfig();
    auto provider = MakeProviderConfig();
    agent.Provider = provider.Id = "provider";

    Db_.Agents()->InsertOrUpdate(agent).GetValueSync();
    Db_.Providers()->Insert(provider).GetValueSync();

    ActorSystem_->Send(UpdaterId_, MakeHolder<TEvents::TEvWakeup>());
    TEvConfigChanged::TPtr ev;
    do {
        ev = ActorSystem_->GrabEdgeEvent<TEvConfigChanged>(TestActorId_);
    } while (ev->Get()->Added.empty());

    auto second = agent;
    second.Provider = agent.Provider;
    second.Hostname = "hostname2";
    Db_.Agents()->InsertOrUpdate(second).GetValueSync();
    ActorSystem_->Send(UpdaterId_, MakeHolder<TEvents::TEvWakeup>());
    do {
        ev = ActorSystem_->GrabEdgeEvent<TEvConfigChanged>(TestActorId_);
    } while (ev->Get()->Changed.empty());

    auto& changed = ev->Get()->Changed;
    auto it = FindIf(changed.begin(), changed.end(), [&] (auto&& s) {
        return s.Type() == EFetcherShardType::Agent;
    });

    ASSERT_THAT(it, Not(Eq(changed.end())));
}

TEST_F(TShardUpdaterTest, AgentDeletion) {
    Y_UNUSED(ActorSystem_->GrabEdgeEvent<TEvConfigChanged>(TestActorId_));
    auto agent1 = MakeAgentConfig();
    auto agent2 = MakeAgentConfig();
    auto provider1 = MakeProviderConfig();
    auto provider2 = MakeProviderConfig();

    agent1.Provider = provider1.Id = "provider1";
    agent2.Provider = provider2.Id = "provider2";

    Db_.Agents()->InsertOrUpdate(agent1).GetValueSync();
    Db_.Agents()->InsertOrUpdate(agent2).GetValueSync();
    Db_.Providers()->Insert(provider1).GetValueSync();
    Db_.Providers()->Insert(provider2).GetValueSync();

    ActorSystem_->Send(UpdaterId_, MakeHolder<TEvents::TEvWakeup>());
    TEvConfigChanged::TPtr ev;
    do {
        ev = ActorSystem_->GrabEdgeEvent<TEvConfigChanged>(TestActorId_);
    } while (ev->Get()->Added.empty());

    {
        Db_.Agents()->DeleteByProvider(agent1.Provider).GetValueSync();
        ActorSystem_->Send(UpdaterId_, MakeHolder<TEvents::TEvWakeup>());

        do {
            ev = ActorSystem_->GrabEdgeEvent<TEvConfigChanged>(TestActorId_);
        } while (ev->Get()->Removed.empty());

        auto& removed = ev->Get()->Removed;
        ASSERT_THAT(removed, SizeIs(1));
        ASSERT_THAT(removed[0].Id.StrId(), HasSubstr(provider1.Id));
        ASSERT_THAT(removed[0].Type, Eq(EFetcherShardType::Agent));
    }

    {
        Db_.Providers()->Delete(provider2.Id).GetValueSync();
        ActorSystem_->Send(UpdaterId_, MakeHolder<TEvents::TEvWakeup>());

        do {
            ev = ActorSystem_->GrabEdgeEvent<TEvConfigChanged>(TestActorId_);
        } while (ev->Get()->Removed.empty());

        auto& removed = ev->Get()->Removed;
        ASSERT_THAT(removed, SizeIs(1));
        ASSERT_THAT(removed[0].Id.StrId(), HasSubstr(provider2.Id));
        ASSERT_THAT(removed[0].Type, Eq(EFetcherShardType::Agent));
    }
}

class TShardUpdaterInitTest: public TShardUpdaterTest {
public:
    void SetUp() override {
        ActorSystem_ = MakeActorSystem(true);
        Db_.Reset();
        FillDatabase();
    }

    void InitBase(IActor* shardMapBuilder) {
        ActorSystem_->AddLocalService(MakeShardMapBuilderId(), {shardMapBuilder, TMailboxType::Simple, 0});
        ActorSystem_->Initialize();

        auto serviceDao = Db_.Services();
        auto shardDao = Db_.Shards();
        auto clusterDao = Db_.Clusters();
        auto agentDao = Db_.Agents();
        auto providerDao = Db_.Providers();

        auto* shardUpdater = CreateConfigUpdaterActor(TConfigUpdaterConfig{
            .ServiceDao = serviceDao,
            .ShardDao = shardDao,
            .ClusterDao = clusterDao,
            .AgentDao = agentDao,
            .ProviderDao = providerDao,
            .Interval = TDuration::Seconds(5),
            .MetricRegistry = MetricRegistry_,
            .Mode = EUpdaterMode::All,
        });

        UpdaterId_ = ActorSystem_->Register(shardUpdater);
        TestActorId_ = ActorSystem_->AllocateEdgeActor();
        ActorSystem_->Send(new IEventHandle{UpdaterId_, TestActorId_, new TEvents::TEvSubscribe});
        ActorSystem_->GrabEdgeEvent<TEvConfigChanged>(TestActorId_);

        // everything is okay as long as this case does not time out
    }
};

TEST_F(TShardUpdaterInitTest, RetriesErrorShardMapLoad) {
    InitBase(new TFailingInitBuilder{2, LocalShardIds_, true});
}

TEST_F(TShardUpdaterInitTest, LongInitialization) {
    InitBase(new TFailingInitBuilder{2, LocalShardIds_, false});
}

class TShardUpdaterRemoveMissingShardsTest: public TShardUpdaterTest {
public:
    void SetUp() override {
        ActorSystem_ = MakeActorSystem(true);
        Db_.Reset();
        FillDatabase();
        InitUpdater();
    }

    void InitUpdater() {
        ActorSystem_->Initialize();

        auto serviceDao = Db_.Services();
        auto shardDao = Db_.Shards();
        auto clusterDao = Db_.Clusters();
        auto agentDao = Db_.Agents();
        auto providerDao = Db_.Providers();

        auto* shardUpdater = CreateConfigUpdaterActor(TConfigUpdaterConfig{
                .ServiceDao = serviceDao,
                .ShardDao = shardDao,
                .ClusterDao = clusterDao,
                .AgentDao = agentDao,
                .ProviderDao = providerDao,
                .Interval = TDuration::Seconds(5),
                .MetricRegistry = MetricRegistry_,
                .Mode = EUpdaterMode::All,
        });

        UpdaterId_ = ActorSystem_->Register(shardUpdater);
        TestActorId_ = ActorSystem_->AllocateEdgeActor();
    }
};

TEST_F(TShardUpdaterRemoveMissingShardsTest, RemoveMissingShards) {
    auto loc = NSolomon::TClusterNode{ FQDNHostName(), 0, 0 };

    ActorSystem_->Send(new IEventHandle{UpdaterId_, TestActorId_, new TEvents::TEvSubscribe});
    // first of all, we should wait for shard addition, but we cannot know what will fire first:
    //  configs update or shard map loading -- so we'll wait for both
    auto ev1 = ActorSystem_->GrabEdgeEvent<TEvConfigChanged>(TestActorId_);
    auto ev2 = ActorSystem_->GrabEdgeEvent<TEvConfigChanged>(TestActorId_);

    auto added1 = ev1->Get()->Added;
    auto added2 = ev2->Get()->Added;

    ASSERT_FALSE(added1.empty() && added2.empty()) << "shards are added";

    // then make sure that the state is determined
    ActorSystem_->Send(new IEventHandle{
            UpdaterId_, TestActorId_, new TEvLoadShardMapResponse{
                    TLoadShardMapResult::FromValue(TVector<TShardAssignment>{ { loc, { 1, 2, 3 }} })
            }
    });
    ActorSystem_->GrabEdgeEvent<TEvConfigChanged>(TestActorId_);

    // now let's change the state
    auto locNew = NSolomon::TClusterNode{ FQDNHostName() + "-new", 1, 0 };
    auto value = TLoadShardMapResult::FromValue(
            TVector<TShardAssignment>{
                    TShardAssignment{ loc, { 2 } },
                    TShardAssignment{ locNew, { 3 } },
            });
    ActorSystem_->Send(new IEventHandle{
            UpdaterId_, TestActorId_, new TEvLoadShardMapResponse{ std::move(value) }
    });

    auto ev = ActorSystem_->GrabEdgeEvent<TEvConfigChanged>(TestActorId_);
    auto removed = ev->Get()->Removed;

    ASSERT_TRUE(!removed.empty()) << "Removed arr should be non-empty";
    ASSERT_THAT(removed.size(), Eq(1ul));
    ASSERT_THAT(removed[0].Id.NumId(), Eq(1ul));
    ASSERT_THAT(removed[0].Type, Eq(EFetcherShardType::Simple));

    auto changed = ev->Get()->Changed;
    ASSERT_THAT(changed.size(), Eq(1ul));
    ASSERT_THAT(changed[0].Id().NumId(), Eq(3ul));

    ASSERT_THAT(changed[0].Location(), Eq(locNew));

    // now let's bring the old shard back
    ActorSystem_->Send(new IEventHandle{
            UpdaterId_, TestActorId_, new TEvLoadShardMapResponse{
                    TLoadShardMapResult::FromValue(TVector<TShardAssignment>{
                            TShardAssignment{ loc, { 1, 2 } },
                            TShardAssignment{ locNew, { 3 } },
                    })
            }
    });

    {
        auto lastEvent = ActorSystem_->GrabEdgeEvent<TEvConfigChanged>(TestActorId_);
        auto lastChanged = lastEvent->Get()->Changed;

        ASSERT_TRUE(!lastChanged.empty());
        ASSERT_THAT(lastChanged.size(), Eq(1ul));
        ASSERT_THAT(lastChanged[0].Id().NumId(), Eq(1ul));
        ASSERT_THAT(lastChanged[0].Type(), Eq(EFetcherShardType::Simple));
        ASSERT_THAT(lastChanged[0].Location(), Eq(loc));
    }
}
