#include "counters.h"

#include <library/cpp/monlib/metrics/fake.h>
#include <library/cpp/monlib/metrics/metric.h>
#include <library/cpp/monlib/metrics/metric_sub_registry.h>

using namespace NMonitoring;

namespace NSolomon::NFetcher {
    class TCounterHolder {
    public:
        static std::shared_ptr<TCounterHolder> ForTotal(IMetricRegistry& registry) {
            TMetricSubRegistry subRegistry({{"projectId", "total"}, {"shardId", "total"}}, &registry);
            auto metrics = std::make_shared<TCounterHolder>();
            metrics->InitMetrics(&subRegistry);
            return metrics;
        }

        static std::shared_ptr<TCounterHolder> ForProject(IMetricRegistry& registry, TStringBuf projectId) {
            TMetricSubRegistry subRegistry({{"projectId", projectId}, {"shardId", "total"}}, &registry);
            auto metrics = std::make_shared<TCounterHolder>();
            metrics->InitMetrics(&subRegistry);
            return metrics;
        }

        static std::shared_ptr<TCounterHolder> ForShard(TStringBuf projectId, TStringBuf shardId, IMetricRegistry& registry) {
            TMetricSubRegistry subRegistry({{"projectId", projectId}, {"shardId", shardId}}, &registry);
            auto metrics = std::make_shared<TCounterHolder>();
            metrics->InitMetrics(&subRegistry);
            return metrics;
        }

        void InitMetrics(IMetricRegistry* metrics) {
            Inflight = metrics->IntGauge(MakeLabels({{"sensor", "pushesInFlight"}}));
            ResponseTime = metrics->HistogramRate(
                    MakeLabels({{"sensor", "pushTimeMillis"}}),
                    ExponentialHistogram(13, 2, 1));
            MetricsWritten = metrics->Rate(MakeLabels({{"sensor", "coremonSink.metricsWritten"}}));
            Success = metrics->Rate(MakeLabels({{"sensor", "coremonSink.success"}}));
            Failure = metrics->Rate(MakeLabels({{"sensor", "coremonSink.failure"}}));
            ProcessingRate = metrics->Gauge(MakeLabels({{"sensor", "coremonSink.processingRate"}}));
            MaxInflight = metrics->IntGauge(MakeLabels({{"sensor", "pushesInFlightMax"}}));
            UrlQueueOverflow = metrics->Rate(MakeLabels({{"sensor", "urlQueueOverflow"}}));
            QueueSize = metrics->IntGauge(MakeLabels({{"sensor", "coremonSink.queueSize"}}));
            QueueMemEstimate = metrics->IntGauge(MakeLabels({{"sensor", "coremonSink.queueMemoryEstimate"}}));
        }

        std::shared_ptr<TMetricRegistry> Registry;
        IIntGauge* Inflight;
        IHistogram* ResponseTime;
        IRate* MetricsWritten;
        IRate* Success;
        IRate* Failure;
        IGauge* ProcessingRate;
        IIntGauge* MaxInflight;
        IRate* UrlQueueOverflow;
        IIntGauge* QueueSize;
        IIntGauge* QueueMemEstimate;
    };

namespace {

    class TQueueCounters final: public TShardQueue::ICounters {
    public:
        TQueueCounters(IIntGauge* total, IIntGauge* project, IIntGauge* shard) noexcept
            : Total_{total}
            , Project_{project}
            , Shard_{shard}
        {
        }

        void AddSize(i64 val) override {
            Total_->Add(val);
            Project_->Add(val);
            Shard_->Add(val);
        }

        void IncSize() override {
            Total_->Inc();
            Project_->Inc();
            Shard_->Inc();
        }

        void DecSize() override {
            Total_->Dec();
            Project_->Dec();
            Shard_->Dec();
        }

    private:
        IIntGauge* Total_;
        IIntGauge* Project_;
        IIntGauge* Shard_;
    };

    class TAggregatedCounters: public ICoremonMetrics {
    public:
        TAggregatedCounters(
                std::shared_ptr<TCounterHolder> total,
                std::shared_ptr<TCounterHolder> project,
                std::shared_ptr<TCounterHolder> shard)
            : Total_{std::move(total)}
            , Project_{std::move(project)}
            , Shard_{std::move(shard)}
        {
        }

        void RecordResponseTime(i64 value) override {
            Total_->ResponseTime->Record(static_cast<double>(value));
            Project_->ResponseTime->Record(static_cast<double>(value));
            Shard_->ResponseTime->Record(static_cast<double>(value));
        }

        void AddMetricsWritten(ui64 value) override {
            Total_->MetricsWritten->Add(value);
            Project_->MetricsWritten->Add(value);
            Shard_->MetricsWritten->Add(value);
        }

        void Success() override {
            Total_->Success->Inc();
            Project_->Success->Inc();
            Shard_->Success->Inc();
        }

        void Fail() override {
            Total_->Failure->Inc();
            Project_->Failure->Inc();
            Shard_->Failure->Inc();
        }

        void IncInflight() override {
            Total_->Inflight->Inc();
            Project_->Inflight->Inc();
            Shard_->Inflight->Inc();
        }

        ui64 DecInflight() override {
            Total_->Inflight->Dec();
            Project_->Inflight->Dec();
            return Shard_->Inflight->Dec();
        }

        void UrlQueueOverflow() override {
            Total_->UrlQueueOverflow->Inc();
            Project_->UrlQueueOverflow->Inc();
            Shard_->UrlQueueOverflow->Inc();
        }

        TEwmaMeter CreateProcessingRateEwma() override {
            return TEwmaMeter(FiveMinuteEwma(Shard_->ProcessingRate));
        }

        void AddMaxInflight(i64 val) override {
            Total_->MaxInflight->Add(val);
            Project_->MaxInflight->Add(val);
            Shard_->MaxInflight->Add(val);
        }

        THolder<TShardQueue::ICounters> CreateQueueCounters() override {
            return MakeHolder<TQueueCounters>(Total_->QueueSize, Project_->QueueSize, Shard_->QueueSize);
        }

        IIntGauge* QueueMemEstimateCounter() override {
            return Shard_->QueueMemEstimate;
        }

    private:
        std::shared_ptr<TCounterHolder> Total_;
        std::shared_ptr<TCounterHolder> Project_;
        std::shared_ptr<TCounterHolder> Shard_;
    };

    class TOnlyTotalMetrics: public ICoremonMetrics {
    public:
        explicit TOnlyTotalMetrics(std::shared_ptr<TCounterHolder> total) noexcept
            : Total_{std::move(total)}
        {
        }

        void RecordResponseTime(i64 value) override {
            Total_->ResponseTime->Record(static_cast<double>(value));
        }

        void AddMetricsWritten(ui64 value) override {
            Total_->MetricsWritten->Add(value);
        }

        void Success() override {
            Total_->Success->Inc();
        }

        void Fail() override {
            Total_->Success->Inc();
        }

        void IncInflight() override {
            Total_->Inflight->Inc();
        }

        ui64 DecInflight() override {
            Total_->Inflight->Dec();
            return 0;
        }

        void UrlQueueOverflow() override {
            Total_->UrlQueueOverflow->Inc();
        }

        void AddMaxInflight(i64 val) override {
            Total_->MaxInflight->Add(val);
        }

        TEwmaMeter CreateProcessingRateEwma() override {
            return TEwmaMeter(FiveMinuteEwma(&FakeGauge_));
        }

        THolder<TShardQueue::ICounters> CreateQueueCounters() override {
            return CreateFakeCounters();
        }

        IIntGauge* QueueMemEstimateCounter() override {
            return &FakeIntGauge_;
        }

    private:
        std::shared_ptr<TCounterHolder> Total_;
        TFakeGauge FakeGauge_;
        TFakeIntGauge FakeIntGauge_;
    };

} // namespace

    TCoremonMetricFactory::TCoremonMetricFactory(
            std::shared_ptr<IMetricRegistry> shardMetrics,
            EMetricVerbosity verbosity)
        : ShardMetrics_{std::move(shardMetrics)}
        , Total_{TCounterHolder::ForTotal(*ShardMetrics_)}
        , Verbosity_{verbosity}
    {
    }

    ICoremonMetricsPtr TCoremonMetricFactory::MetricsFor(TStringBuf projectId, TStringBuf shardId) {
        switch (Verbosity_) {
            case EMetricVerbosity::All:
                return std::make_shared<TAggregatedCounters>(
                        Total_,
                        GetPerProjectMetrics(projectId),
                        TCounterHolder::ForShard(projectId, shardId, *ShardMetrics_));

            case EMetricVerbosity::OnlyTotal:
                return std::make_shared<TOnlyTotalMetrics>(Total_);
        }

        Y_UNREACHABLE();
    }

    std::shared_ptr<TCounterHolder> TCoremonMetricFactory::GetPerProjectMetrics(TStringBuf projectId) {
        auto& projectCounters = PerProject_[projectId];
        if (!projectCounters) {
            projectCounters = TCounterHolder::ForProject(*ShardMetrics_, projectId);
        }
        return projectCounters;
    }

} // namespace NSolomon::NFetcher
