#include "limiter.h"

#include <library/cpp/monlib/metrics/metric_registry.h>

#include <utility>

using namespace NMonitoring;


namespace NSolomon::NFetcher {
    TGlobalLimiter::TGlobalLimiter(ui64 limit, IIntGauge* memory)
        : Limit_{limit}
        , Memory_{memory}
    {
    }

    bool TGlobalLimiter::OnWrite(ui64 bytes) {
        ui64 newVal;
        ui64 oldVal = Current_.load(std::memory_order_relaxed);

        do {
            if (oldVal + bytes > Limit_) {
                return false;
            }

            newVal = oldVal + bytes;
        } while (!Current_.compare_exchange_weak(oldVal, newVal,
            std::memory_order_release, std::memory_order_consume));

        Memory_->Add(bytes);
        return true;
    }

    void TGlobalLimiter::OnFree(ui64 bytes) {
        Current_.fetch_sub(bytes);
        Memory_->Add(-bytes);
    }

    TShardLimiter::TShardLimiter(ui64 shardLimit, IIntGauge* memory, IQueueMemoryLimiterPtr globalLimiter)
        : Limit_{shardLimit}
        , Global_{std::move(globalLimiter)}
        , Memory_{memory}
    {
    }

    TShardLimiter::~TShardLimiter() {
        Global_->OnFree(Current_);
    }

    bool TShardLimiter::OnWrite(ui64 bytes) {
        if (!Global_->OnWrite(bytes)) {
            return false;
        }

        if (Current_ + bytes > Limit_) {
            Global_->OnFree(bytes);
            return false;
        }

        Memory_->Add(bytes);
        Current_ += bytes;
        return true;
    }

    void TShardLimiter::OnFree(ui64 bytes) {
        const auto toFree = Min(bytes, Current_);
        Global_->OnFree(toFree);
        Current_ -= toFree;
        Memory_->Add(-toFree);
    }
} // namespace NSolomon::NFetcher
