#include "processing_client.h"

#include <solomon/libs/cpp/proto_convert/format.h>
#include <solomon/libs/cpp/proto_convert/labels.h>

#include <util/random/random.h>

namespace NSolomon::NFetcher {
namespace {

    using TCoremonResult = TErrorOr<NCoremon::TProcessResult, TApiCallError>;
    using TFetcherResult = TErrorOr<NFetcher::TProcessResult, TApiCallError>;

    class TCoremonRpcClient: public IProcessingClient {
    public:
        explicit TCoremonRpcClient(NCoremon::ICoremonClient* impl)
            : Impl_{impl}
        {
        }

        TAsyncDataProcessResponse ProcessPulledData(
            const TShardId& shardId,
            std::vector<TQueueEntry> entries) override
        {
            // TODO: right now there will be only one queue entry,
            //       after Coremon will be prepared to receive many documents in single request,
            //       then we can send them together
            std::vector<NThreading::TFuture<TCoremonResult>> futures;
            futures.reserve(entries.size());

            for (auto& entry: entries) {
                futures.push_back(Impl_->ProcessPulledData(
                    shardId.NumId(),
                    std::move(entry.Host),
                    entry.Labels,
                    entry.Format,
                    entry.Instant,
                    std::move(entry.Data),
                    entry.PrevInstant,
                    std::move(entry.PrevData)
                ));
            }

            return NThreading::WaitAll(futures)
                .Apply([futures = std::move(futures)] (auto) {
                    std::vector<TFetcherResult> results;
                    results.reserve(futures.size());

                    for (auto&& f: futures) {
                        Y_VERIFY(f.HasValue() || f.HasException());

                        try {
                            const auto& result = f.GetValue();
                            if (result.Fail()) {
                                results.push_back(result.PassError<NFetcher::TProcessResult>());
                            } else {
                                const auto& value = result.Value();
                                results.emplace_back(TProcessResult{
                                    .SuccessMetricCount = value.SuccessMetricCount,
                                    .Status = value.Status,
                                    .Error = value.Error,
                                });
                            }
                        } catch (...) {
                            TString errorMessage = CurrentExceptionMessage();
                            results.push_back(TFetcherResult::FromError(TApiCallError{errorMessage}));
                        }
                    }

                    return results;
                });
        }

        TAsyncShardCreated CreateShard(
            TString projectId,
            TString serviceName,
            TString clusterName,
            TString createdBy) noexcept override
        {
            return Impl_->CreateShard(
                    std::move(projectId),
                    std::move(serviceName),
                    std::move(clusterName),
                    std::move(createdBy))
                .Apply([] (auto f) -> TErrorOr<TShardCreated, TApiCallError> {
                    auto result = f.ExtractValue();
                    if (result.Fail()) {
                        return result.template PassError<TShardCreated>();
                    }

                    auto v = result.Extract();
                    return TShardCreated{
                        .Leader = v.Leader,
                        .AssignedToHost = v.AssignedToHost,
                        .ShardId = v.ShardId,
                        .NumId = v.NumId,
                    };
                });
        }

        TAsyncShardAssignments GetShardAssignments() noexcept override {
            return Impl_->GetShardAssignments()
                .Apply([] (auto f)  -> TErrorOr<TShardAssignments, TApiCallError> {
                    auto result = f.ExtractValue();
                    if (result.Fail()) {
                        return result.template PassError<TShardAssignments>();
                    }

                    auto assn = result.Extract();
                    return TShardAssignments{
                        .Leader = std::move(assn.Leader),
                        .Assignments = std::move(assn.Assignments)
                    };
                });
        }

        void Stop(bool wait) override {
            Impl_->Stop(wait);
        }

        bool IsBatchSupported() const noexcept override {
            return false;
        }

    private:
        NCoremon::ICoremonClient* Impl_;
    };

    class TIngestorRpcClient: public IProcessingClient {
    public:
        explicit TIngestorRpcClient(NIngestor::IIngestorClient* impl)
            : Impl_{impl}
        {
        }

        TAsyncDataProcessResponse ProcessPulledData(
            const TShardId& shardId,
            std::vector<TQueueEntry> entries) noexcept override
        {
            size_t size = entries.size();

            yandex::monitoring::ingestor::TPulledDataRequest req;
            req.SetNumId(shardId.NumId());

            for (auto& entry: entries) {
                req.AddHost(std::move(entry.Host));
                req.AddFormat(ConvertFormatToProto(entry.Format));
                req.AddSourceId(entry.SourceId);
                WriteLabelsToProto(*req.AddOptionalLabels(), entry.Labels);
                req.AddResponse(std::move(entry.Data));
                req.AddResponseTimeMillis(entry.Instant.MilliSeconds());
            }

            return Impl_->ProcessPulledData(req).Apply([size] (auto f) -> std::vector<TFetcherResult> {
                try {
                    auto result = f.ExtractValue();
                    if (result.Fail()) {
                        std::vector<TFetcherResult> results;
                        results.reserve(size);
                        for (size_t i = 0; i < size; i++) {
                            results.emplace_back(result.template PassError<NFetcher::TProcessResult>());
                        }
                        return results;
                    }

                    const auto& val = result.Value();
                    size_t resultSize = static_cast<size_t>(val.status_size());

                    std::vector<TFetcherResult> results;
                    results.reserve(resultSize);

                    for (size_t i = 0; i < resultSize; i++) {
                        results.emplace_back(TProcessResult{
                            .SuccessMetricCount = val.GetSuccessMetricCount(i),
                            .Status = val.GetStatus(i),
                            .Error = std::move(val.GetErrorMessage(i)),
                        });
                    }

                    return results;
                } catch (...) {
                    TApiCallError error{CurrentExceptionMessage()};
                    std::vector<TFetcherResult> results;
                    results.reserve(size);
                    for (size_t i = 0; i < size; i++) {
                        results.emplace_back(TFetcherResult::FromError(error));
                    }
                    return results;
                }
            });
        }

        TAsyncShardCreated CreateShard(
            TString projectId,
            TString serviceName,
            TString clusterName,
            TString createdBy) noexcept override
        {
            yandex::monitoring::ingestor::TCreateShardRequest req;
            req.SetProjectId(std::move(projectId));
            req.SetServiceName(std::move(serviceName));
            req.SetClusterName(std::move(clusterName));
            req.SetCreatedBy(std::move(createdBy));

            return Impl_->CreateShard(req)
                .Apply([] (auto f) -> TErrorOr<TShardCreated, TApiCallError> {
                    auto result = f.ExtractValue();
                    if (result.Fail()) {
                        return result.template PassError<TShardCreated>();
                    }

                    auto v = result.Extract();
                    return TShardCreated{
                        .Leader = v.Leader,
                        .AssignedToHost = v.AssignedToHost,
                        .ShardId = v.ShardId,
                        .NumId = v.NumId,
                    };
                });
        }

        TAsyncShardAssignments GetShardAssignments() noexcept override {
            using TResult = NThreading::TFutureType<TAsyncShardAssignments>;

            TString msg{"method GetShardAssignments() is not implemented for Ingestor"};
            auto status = ::NGrpc::TGrpcStatus::Internal(msg);
            return NThreading::MakeFuture(TResult::FromError(std::move(status), std::move(msg)));
        }

        void Stop(bool wait) override {
            Impl_->Stop(wait);
        }

        bool IsBatchSupported() const noexcept override {
            return true;
        }

    private:
        NIngestor::IIngestorClient* Impl_;
    };

    template <typename TImpl>
    class TProcessingClusterClient: public IProcessingClusterClient {
    public:
        TProcessingClusterClient(TImpl impl)
            : Impl_{std::move(impl)}
        {
            const auto& addresses = Impl_->Addresses();

            for (const auto& addr: addresses) {
                auto client = Impl_->Get(TString{addr});
                Clients_.try_emplace(addr, WrapProcessingClient(client));
            }
        }

        IProcessingClientPtr GetClient(const TClusterNode& node) noexcept override {
            if (auto it = Clients_.find(node.Endpoint); it != Clients_.end()) {
                return it->second;
            }

            return nullptr;
        }

        IProcessingClientPtr GetAnyClient() override {
            auto it = Clients_.begin();
            for (size_t n = RandomNumber(Clients_.size()); n; --n) {
                ++it;
            }
            Y_VERIFY(it != Clients_.end());
            return it->second;
        }

    private:
        TImpl Impl_;
        THashMap<TString, IProcessingClientPtr> Clients_;
    };
} // namespace

    IProcessingClientPtr WrapProcessingClient(NCoremon::ICoremonClient* coremonClient) {
        return MakeIntrusive<TCoremonRpcClient>(coremonClient);
    }

    IProcessingClientPtr WrapProcessingClient(NIngestor::IIngestorClient* ingestorClient) {
        return MakeIntrusive<TIngestorRpcClient>(ingestorClient);
    }

    IProcessingClusterClientPtr WrapClusterClient(NCoremon::ICoremonClusterClientPtr cluster) {
        return new TProcessingClusterClient<NCoremon::ICoremonClusterClientPtr>{std::move(cluster)};
    }

    IProcessingClusterClientPtr WrapClusterClient(NIngestor::IIngestorClusterClientPtr cluster) {
        return new TProcessingClusterClient<NIngestor::IIngestorClusterClientPtr>{std::move(cluster)};
    }
} // namespace NSolomon::NFetcher
