#include "shard_writer.h"
#include "limiter.h"
#include "counters.h"

#include <solomon/services/fetcher/lib/sink/sink.h>

#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/error_or/error_or.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/event_local.h>
#include <library/cpp/actors/core/hfunc.h>

#include <util/system/hp_timer.h>

using namespace NActors;
using namespace NMonitoring;
using namespace yandex::solomon::common;

namespace NSolomon::NFetcher {
namespace {
    bool IsDataProcessed(const TErrorOr<TProcessResult, TApiCallError>& result) {
        if (result.Fail()) {
            return true;
        }

        auto status = result.Value().Status;
        switch (status) {
            case CONNECT_FAILURE:
            case SHARD_NOT_INITIALIZED:
            case UNKNOWN_PROJECT:
            case SHARD_IS_NOT_WRITABLE:
            case UNKNOWN_SHARD:
                return false;

            default:
                break;
        }

        return true;
    }

    UrlStatusType ClassifyError(const TApiCallError& err) {
        if (!err.IsTransportError()) {
            return UNKNOWN_ERROR;
        }

        auto grpcStatus = err.Status();
        switch (grpcStatus) {
            case grpc::StatusCode::DEADLINE_EXCEEDED:
                return TIMEOUT;
            case grpc::StatusCode::RESOURCE_EXHAUSTED:
            case grpc::StatusCode::UNAVAILABLE:
            case grpc::StatusCode::CANCELLED:
                return IPC_QUEUE_OVERFLOW;
            default:
                return UNKNOWN_ERROR;
        }
    }

    constexpr i64 MIN_IN_FLIGHT = 10;
    constexpr size_t MAX_BATCH_SIZE = 80 << 20; // 80 MiB
    constexpr size_t MAX_URL_QUEUE = 20;
    constexpr size_t MAX_SHARD_QUEUE = 0; // no limit

    class TShardWriter: public TActor<TShardWriter>, private TPrivateEvents {
        enum {
            WriteDone = SpaceBegin,
            End,
        };
        static_assert(End < SpaceEnd, "too many event types");

        struct TRequestContext {
            ui64 Interval{0};
            TDuration RequestDuration;
            TActorId SenderId;
            ui64 Size;
        };

        struct TWriteDone: public TEventLocal<TWriteDone, WriteDone> {
            TWriteDone(TErrorOr<TProcessResult, TApiCallError>&& value, const TRequestContext& ctx)
                : Value{std::move(value)}
                , Ctx{ctx}
            {
            }

            TErrorOr<TProcessResult, TApiCallError> Value;
            TRequestContext Ctx;
        };

    public:
        // NOLINTNEXTLINE(performance-unnecessary-value-param): false positive
        TShardWriter(TShardId shardId, IProcessingClient& client, ICoremonMetricsPtr counters, IQueueMemoryLimiterPtr limiter)
            : TActor<TShardWriter>{&TThis::StateWork}
            , Counters_{std::move(counters)}
            , Queue_{MAX_URL_QUEUE, MAX_SHARD_QUEUE, Counters_->CreateQueueCounters(), std::move(limiter)}
            , Processor_{client}
            , ShardId_{std::move(shardId)}
        {
            ProcessingRate_ = Counters_->CreateProcessingRateEwma();
            MaxInflight_ = 100;
            Counters_->AddMaxInflight(MaxInflight_);
        }

        STATEFN(StateWork) {
            switch (ev->GetTypeRewrite()) {
                hFunc(TShardWriterEvents::TWrite, OnWrite);
                hFunc(TWriteDone, OnWriteDone);
                sFunc(TEvents::TEvPoison, OnPoison);
            }
        }

        STATEFN(StateDying) {
            switch (ev->GetTypeRewrite()) {
                sFunc(TWriteDone, OnDecInflight);
            }
        }

        void OnPoison() {
            // if there are no ongoing requests to coremon, just die; otherwise wait for them to complete
            if (Inflight_ > 0) {
                // ensure all futures are completed
                Become(&TThis::StateDying);
            } else {
                PassAway();
            }
        }

        void OnWrite(const TShardWriterEvents::TWrite::TPtr& ev) {
            const auto url = ev->Get()->Entry.Url;
            const auto sender = ev->Get()->Entry.Sender;

            Y_VERIFY_DEBUG(MaxInflight_ > 0);

            // postpone push in case the number of outstanding requests is too big
            if (Inflight_ < MaxInflight_) {
                Write(std::move(ev->Get()->Entry));
            } else if (!Queue_.Push(std::move(ev->Get()->Entry))) {
                MON_INFO(CoremonSink, "Write to " << url << " rejected due to URL queue overflow");
                Counters_->UrlQueueOverflow();

                Send(sender, new TEvMetricDataWritten{ShardId_.StrId(), IPC_QUEUE_OVERFLOW, 0, "URL queue overflow"});
            }
        }

        ui64 CalcMaxInflight(ui64 interval) {
            constexpr auto MAX_GROWTH_FACTOR = 10;

            const auto prevMaxInflight = MaxInflight_;
            const auto docsPerSecond = ProcessingRate_.Get();
            const auto maxInflight = ui64(docsPerSecond * interval);

            // disallow max inflight to grow too fast, but set it's lower limit
            return Max(Min<ui64>(prevMaxInflight * MAX_GROWTH_FACTOR, maxInflight), ui64(MIN_IN_FLIGHT));
        }

        void OnDecInflight() {
            if (DecInflight() == 0) {
                PassAway();
            }
        }

        void OnWriteDone(const TWriteDone::TPtr& ev) {
            DecInflight();

            const auto senderId = ev->Get()->Ctx.SenderId;
            const auto interval = ev->Get()->Ctx.Interval;

            Counters_->RecordResponseTime(ev->Get()->Ctx.RequestDuration.MilliSeconds());

            const auto prevMax = MaxInflight_;
            MaxInflight_ = CalcMaxInflight(interval);
            Counters_->AddMaxInflight(MaxInflight_ - prevMax);

            auto&& v = ev->Get()->Value;
            std::unique_ptr<TEvMetricDataWritten> result = v.Success()
                    ? ToResult(v.Extract())
                    : ErrorToResult(v.Error());
            Send(senderId, result.release());

            const i64 toTake = MaxInflight_ - Inflight_;
            for (i64 i = 0u; i < toTake && !Queue_.IsEmpty(); ++i) {
                Write(Queue_.Pop());
            }
        }

        std::unique_ptr<TEvMetricDataWritten> ErrorToResult(const TApiCallError& err) {
            Counters_->Fail();
            UrlStatusType status = ClassifyError(err);
            if (status == UNKNOWN_ERROR) {
                MON_ERROR(CoremonSink, "Write to shard " << ShardId_ << " failed: " << err.Message());
            } else {
                MON_INFO(CoremonSink, "Write to shard " << ShardId_ <<
                         " failed: " << UrlStatusType_Name(status) << ' ' << err.Message());
            }
            return std::make_unique<TEvMetricDataWritten>(ShardId_.StrId(), status, 0, err.MessageString());
        }

        std::unique_ptr<TEvMetricDataWritten> ToResult(TProcessResult result) {
            if (result.Status == OK) {
                Counters_->Success();
                Counters_->AddMetricsWritten(result.SuccessMetricCount);
            } else {
                Counters_->Fail();
            }
            return std::make_unique<TEvMetricDataWritten>(
                    ShardId_.StrId(),
                    result.Status,
                    result.SuccessMetricCount,
                    std::move(result.Error));
        }

        void Write(TQueueEntry&& val) {
            auto* as = TActorContext::ActorSystem();
            const auto senderId = val.Sender;
            const auto interval = val.Interval.Seconds();
            const auto self = SelfId();
            const auto size = val.Size();

            MON_DEBUG(CoremonSink, "writing to data sink " << size << " bytes");

            THPTimer timer;
            IncInflight();

            Processor_.ProcessPulledData(ShardId_, std::vector<TQueueEntry>{std::move(val)})
                .Subscribe([=] (auto f) mutable {
                    auto v = f.ExtractValue();

                    if (IsDataProcessed(v[0])) {
                        ProcessingRate_.Mark();
                    }

                    TRequestContext ctx {
                        .Interval = interval,
                        .RequestDuration = TDuration::FromValue(timer.Passed()),
                        .SenderId = senderId,
                        .Size = size,
                    };

                    as->Send(self, new TWriteDone{std::move(v[0]), ctx});
                });
        }

    private:
        i64 IncInflight() {
            Inflight_++;
            Counters_->IncInflight();
            return Inflight_;
        }

        i64 DecInflight() {
            Inflight_--;
            Counters_->DecInflight();
            return Inflight_;
        }

    private:
        ICoremonMetricsPtr Counters_;
        TShardQueue Queue_;

        IProcessingClient& Processor_;
        /**
         * Id of a Simple shard (not Solomon agent shard)
         */
        TShardId ShardId_;

        TEwmaMeter ProcessingRate_;
        i64 Inflight_{0};
        i64 MaxInflight_{MIN_IN_FLIGHT};
    };

class TBatchingShardWriter: public TActor<TBatchingShardWriter>, private TPrivateEvents {
    enum {
        WriteDone = SpaceBegin,
        End,
    };
    static_assert(End < SpaceEnd, "too many event types");

    struct TWriteDone: public TEventLocal<TWriteDone, WriteDone> {
        TDuration WriteDuration;
        std::vector<TActorId> Senders;
        std::vector<TErrorOr<TProcessResult, TApiCallError>> Values;
    };

public:
    // NOLINTNEXTLINE(performance-unnecessary-value-param): false positive
    TBatchingShardWriter(TShardId shardId, IProcessingClient& client, ICoremonMetricsPtr counters, IQueueMemoryLimiterPtr limiter)
        : TActor<TBatchingShardWriter>{&TThis::StateWork}
        , Counters_{std::move(counters)}
        , Queue_{MAX_URL_QUEUE, MAX_SHARD_QUEUE, Counters_->CreateQueueCounters(), std::move(limiter)}
        , Processor_{client}
        , ShardId_{std::move(shardId)}
    {
    }

    STATEFN(StateWork) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TShardWriterEvents::TWrite, OnWrite);
            hFunc(TWriteDone, OnWriteDone);
            sFunc(TEvents::TEvPoison, OnPoison);
        }
    }

    STATEFN(StateDying) {
        switch (ev->GetTypeRewrite()) {
            sFunc(TWriteDone, OnDecInflight);
        }
    }

    void OnPoison() {
        // if there are no ongoing requests, just die; otherwise wait for them to complete
        if (Inflight_ > 0) {
            Become(&TThis::StateDying);
        } else {
            PassAway();
        }
    }

    void OnWrite(const TShardWriterEvents::TWrite::TPtr& ev) {
        const auto url = ev->Get()->Entry.Url;
        const auto sender = ev->Get()->Entry.Sender;

        if (!Queue_.Push(std::move(ev->Get()->Entry))) {
            MON_INFO(CoremonSink, "Write to " << url << " rejected due to URL queue overflow");
            Counters_->UrlQueueOverflow();
            Send(sender, new TEvMetricDataWritten{ShardId_.StrId(), IPC_QUEUE_OVERFLOW, 0, "URL queue overflow"});
        }

        if (Inflight_ < MIN_IN_FLIGHT) {
            SendBatchWrite();
        }
    }

    void OnDecInflight() {
        if (DecInflight() == 0) {
            PassAway();
        }
    }

    void OnWriteDone(const TWriteDone::TPtr& ev) {
        DecInflight();
        Counters_->RecordResponseTime(ev->Get()->WriteDuration.MilliSeconds());

        auto& senders = ev->Get()->Senders;
        auto& values = ev->Get()->Values;

        if (senders.size() != values.size()) {
            for (size_t i = 0; i < senders.size(); ++i) {
                Send(senders[i],
                     new TEvMetricDataWritten{ShardId_.StrId(), UrlStatusType::UNKNOWN_STATUS_TYPE, 0, "no response"});
            }
        } else {
            for (size_t i = 0; i < values.size(); i++) {
                auto result = values[i].Success() ? ToResult(values[i].Extract()) : ErrorToResult(values[i].Error());
                Send(senders[i], result.release());
            }
        }

        if (Inflight_ < MIN_IN_FLIGHT) {
            SendBatchWrite();
        }
    }

    std::unique_ptr<TEvMetricDataWritten> ErrorToResult(const TApiCallError& err) {
        Counters_->Fail();
        UrlStatusType status = ClassifyError(err);
        if (status == UNKNOWN_ERROR) {
            MON_ERROR(CoremonSink, "Write to shard " << ShardId_ << " failed: " << err.Message());
        } else {
            MON_INFO(CoremonSink, "Write to shard " << ShardId_ <<
                    " failed: " << UrlStatusType_Name(status) << ' ' << err.Message());
        }
        return std::make_unique<TEvMetricDataWritten>(ShardId_.StrId(), status, 0, err.MessageString());
    }

    std::unique_ptr<TEvMetricDataWritten> ToResult(TProcessResult result) {
        if (result.Status == OK) {
            Counters_->Success();
            Counters_->AddMetricsWritten(result.SuccessMetricCount);
        } else {
            Counters_->Fail();
        }
        return std::make_unique<TEvMetricDataWritten>(
                ShardId_.StrId(),
                result.Status,
                result.SuccessMetricCount,
                std::move(result.Error));
    }

    void SendBatchWrite() {
        size_t sizeBytes = 0;
        std::vector<TQueueEntry> entries;
        while (!Queue_.IsEmpty()) {
            sizeBytes += entries.emplace_back(Queue_.Pop()).Size();
            if (sizeBytes >= MAX_BATCH_SIZE) {
                break;
            }
        }

        if (entries.empty()) {
            return;
        }

        MON_DEBUG(CoremonSink, "writing to data sink " << entries.size() << " entries with " << sizeBytes << " bytes");

        THPTimer timer;
        IncInflight();

        auto* actorSystem = TActorContext::ActorSystem();
        auto selfId = SelfId();

        std::vector<TActorId> senders;
        senders.reserve(entries.size());
        for (auto& e: entries) {
            senders.push_back(e.Sender);
        }

        Processor_.ProcessPulledData(ShardId_, std::move(entries))
            .Subscribe([=, senders = std::move(senders)] (auto f) mutable {
                auto result = std::make_unique<TWriteDone>();
                result->Values = f.ExtractValue();
                result->Senders = std::move(senders);
                result->WriteDuration = TDuration::FromValue(timer.Passed());
                actorSystem->Send(selfId, result.release());
            });
    }

private:
    i64 IncInflight() {
        Inflight_++;
        Counters_->IncInflight();
        return Inflight_;
    }

    i64 DecInflight() {
        Inflight_--;
        Counters_->DecInflight();
        return Inflight_;
    }

private:
    ICoremonMetricsPtr Counters_;
    TShardQueue Queue_;

    IProcessingClient& Processor_;
    /**
     * Id of a Simple shard (not Solomon agent shard)
     */
    TShardId ShardId_;
    i64 Inflight_{0};
};

} // namespace

IActor* CreateShardWriter(TShardId shardId, IProcessingClient& client, ICoremonMetricsPtr counters, IQueueMemoryLimiterPtr limiter) {
    if (client.IsBatchSupported()) {
        return new TBatchingShardWriter{std::move(shardId), client, std::move(counters), std::move(limiter)};
    }
    return new TShardWriter{std::move(shardId), client, std::move(counters), std::move(limiter)};
}

} // namespace NSolomon::NFetcher
