#include "data_sink.h"
#include "limiter.h"
#include "counters.h"
#include "shard_writer.h"

#include <solomon/services/fetcher/lib/id.h>

#include <solomon/libs/cpp/http/server/handlers/metrics.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>

using namespace NActors;
using namespace NMonitoring;

namespace NSolomon::NFetcher {
namespace {
    class TShardWriterFactory: public IShardWriterFactory {
    public:
        TShardWriterFactory(IProcessingClientPtr client, TCoremonSinkConf conf)
            : ShardMetrics_{std::move(conf.ShardMetrics)}
            , Client_{std::move(client)}
            , MetricFactory_{MakeHolder<TCoremonMetricFactory>(ShardMetrics_, conf.MetricVerbosity)}
            , GlobalLimiter_{MakeIntrusive<TGlobalLimiter>(
                    static_cast<ui64>(conf.MemLimit),
                    ShardMetrics_->IntGauge(MakeLabels({
                        {"sensor", "coremonSink.queueMemoryEstimate"},
                        {"shardId", "total"},
                        {"projectId", "total"}
                    }))
            )}
            , MemoryLimit_{conf.MemLimit}
        {
        }

        IActor* CreateShardWriter(TStringBuf projectId, TShardId shardId) override {
            auto metrics = MetricFactory_->MetricsFor(projectId, shardId.StrId());

            auto limiter = CreateShardLimiter(metrics->QueueMemEstimateCounter());
            return NSolomon::NFetcher::CreateShardWriter(
                shardId,
                *Client_,
                std::move(metrics),
                std::move(limiter)
            );
        }

        TIntrusivePtr<TGlobalLimiter> Limiter() const override {
            return GlobalLimiter_;
        }

    private:
        IQueueMemoryLimiterPtr CreateShardLimiter(IIntGauge* memEstimate) {
            // allow single shard to consume only up to 20% of the allocated memory
            // to prevent monster shards to eventually use up all the memory

            return new TShardLimiter{
                ui64(MemoryLimit_) / 5,
                memEstimate,
                GlobalLimiter_,
            };
        }

    private:
        std::shared_ptr<IMetricRegistry> ShardMetrics_;
        IProcessingClientPtr Client_;
        THolder<TCoremonMetricFactory> MetricFactory_;
        TIntrusivePtr<TGlobalLimiter> GlobalLimiter_;
        i64 MemoryLimit_{};
    };
} // namespace
    IShardWriterFactoryPtr CreateShardWriterFactory(IProcessingClientPtr client, TCoremonSinkConf conf) {
        return new TShardWriterFactory{std::move(client), std::move(conf)};
    }
} // namespace NSolomon::NFetcher
