#include "data_sink.h"
#include "limiter.h"
#include "shard_writer.h"

#include <solomon/services/fetcher/lib/app_data.h>
#include <solomon/services/fetcher/lib/config_updater/config_updater.h>
#include <solomon/services/fetcher/lib/sink/sink.h>
#include <solomon/services/fetcher/lib/id.h>

#include <solomon/libs/cpp/logging/logging.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/event_local.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/actors/core/log.h>

#include <library/cpp/monlib/metrics/metric.h>

#include <util/generic/size_literals.h>

using namespace NActors;
using namespace NMonitoring;

namespace NSolomon::NFetcher {
namespace {
    class TSinkActor: public TActorBootstrapped<TSinkActor> {
        struct TShard {
            TShard(TActorId actorId, TString projectId)
                : ActorId{actorId}
                , ProjectId{std::move(projectId)}
            {
            }

            TActorId ActorId;
            TString ProjectId;
        };

    public:
        TSinkActor(IShardWriterFactoryPtr writerFactory, ICounter* overflows, ui64 executorPool)
            : Overflows_{overflows}
            , ExecutorPool_{executorPool}
            , ShardWriterFactory_{std::move(writerFactory)}
            , GlobalLimiter_{ShardWriterFactory_->Limiter()}
        {
        }

        void Bootstrap() {
            Send(MakeConfigUpdaterId(), new TEvents::TEvSubscribe);
            Become(&TThis::StateWork);
        }

        STATEFN(StateWork) {
            switch (ev->GetTypeRewrite()) {
                hFunc(TEvSinkWrite, OnMetricData);
                hFunc(TEvConfigChanged, OnConfigChanged);
                cFunc(TEvents::TSystem::PoisonPill, OnPoison);
            }
        }

        void OnConfigChanged(const TEvConfigChanged::TPtr& ev) {
            for (auto&& shardInfo: ev->Get()->Removed) {
                RemoveShard(shardInfo.Id);
            }

            for (auto&& shard: ev->Get()->Changed) {
                if (!shard.IsLocal()) {
                    RemoveShard(shard.Id());
                }
            }
        }

        void RemoveShard(const TShardId& shardId) {
            auto it = Shards_.find(shardId);
            if (it == Shards_.end()) {
                return;
            }

            Send(it->second.ActorId, new TEvents::TEvPoisonPill);
            Shards_.erase(it);
        }

        void OnPoison() {
            for (auto&& [id, s]: Shards_) {
                Y_UNUSED(id);
                Send(s.ActorId, new TEvents::TEvPoisonPill);
            }

            PassAway();
        }

        void OnMetricData(const TEvSinkWrite::TPtr& evPtr) {
            auto& ev = *evPtr->Get();
            auto& data = ev.ShardData;

            auto dataSize = data.Size();
            if (GlobalLimiter_->Available() < dataSize) {
                MON_ERROR(CoremonSink, "Write to shard " << data.ShardId << " rejected due to memory overflow");
                Send(evPtr->Sender, new TEvMetricDataWritten{
                        data.ShardId.StrId(),
                        yandex::solomon::common::IPC_QUEUE_OVERFLOW,
                        0,
                        TString{"Queue memory limit exceeded"}});
                Overflows_->Inc();
                return;
            }

            DispatchWrite(evPtr->Sender, std::move(ev));
        }

    private:
        void DispatchWrite(TActorId senderId, TEvSinkWrite&& ev) {
            auto&& shardId = ev.ShardData.ShardId;
            auto&& projectId = ev.ShardData.ProjectId;
            auto it = Shards_.find(shardId);

            if (it == Shards_.end()) {
                const auto aid = Register(
                    ShardWriterFactory_->CreateShardWriter(projectId, shardId),
                    TMailboxType::HTSwap,
                    ExecutorPool_
                );

                std::tie(it, std::ignore) = Shards_.emplace(
                    std::piecewise_construct,
                    std::forward_as_tuple(shardId),
                    std::forward_as_tuple(aid, projectId));
            }

            Send(it->second.ActorId, new TShardWriterEvents::TWrite{std::move(ev), senderId});
        }

    private:
        /**
         * Simple shards (not Solomon Agent shards)
         */
        THashMap<TShardId, TShard> Shards_;
        ICounter* Overflows_{nullptr};
        ui64 ExecutorPool_{0};
        IShardWriterFactoryPtr ShardWriterFactory_;
        TIntrusivePtr<TGlobalLimiter> GlobalLimiter_;
    };
} // namespace

NActors::IActor* CreateDataSink(
    IShardWriterFactoryPtr writerFactory,
    ICounter* overflows,
    ui64 executorPool)
{
    return new TSinkActor{std::move(writerFactory), overflows, executorPool};
}

} // namespace NSolomon::NFetcher
