#include "continuous_resolver.h"

#include <solomon/services/fetcher/lib/racktables/racktables_actor.h>

#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/string_map/string_map.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>

#include <library/cpp/containers/absl_flat_hash/flat_hash_set.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

using namespace NActors;
using namespace NThreading;
using namespace NMonitoring;

namespace NSolomon::NFetcher {
namespace {
    constexpr auto MAX_IN_FLIGHT = 2048;

    struct TTask {
        explicit TTask(TString hostname) noexcept
            : Hostname{std::move(hostname)}
        {
        }

        TDuration Duration() const {
            return LastRequestCompleted - LastRequestStarted;
        }

        const TString Hostname;
        TIpv6Address KnownAddress;
        EDc KnownDc{EDc::UNKNOWN};
        absl::flat_hash_set<TActorId, THash<TActorId>> Listeners;
        TInstant LastRequestStarted;
        TInstant LastRequestCompleted;
    };

    class TContinuousResolverActor: public TActorBootstrapped<TContinuousResolverActor>, private TPrivateEvents {
        enum {
            EvPostponed = SpaceBegin,
            EvResolveDone,
            EvResolveFail,
            End,
        };
        static_assert(End < SpaceEnd, "too many event types");

        struct TEvPostponed: TEventLocal<TEvPostponed, EvPostponed> {
            explicit TEvPostponed(TTask* task) noexcept
                : Task{task}
            {
            }

            TTask* Task;
        };

        struct TEvResolveDone: TEventLocal<TEvResolveDone, EvResolveDone> {
            TEvResolveDone(TTask* task, TIpv6Address address) noexcept
                : Task{task}
                , Address{std::move(address)}
            {
            }

            TTask* Task;
            TIpv6Address Address;
        };

        struct TEvResolveFail: TEventLocal<TEvResolveFail, EvResolveFail> {
            TEvResolveFail(TTask* task, EErrorType type, TString reason) noexcept
                : Task{task}
                , Type{type}
                , Reason{std::move(reason)}
            {
            }

            TTask* Task;
            EErrorType Type;
            TString Reason;
        };

    public:
        TContinuousResolverActor(TDnsResolverActorConf conf)
            : WakeupInterval_{conf.WakeupInterval}
            , RefreshInterval_{conf.RefreshInterval}
            , DnsClient_{conf.DnsClient}
            , RackTablesActorId_{conf.RackTablesActorId}
            , Metrics_{
                .InFlight = conf.MetricRegistry.IntGauge({{"sensor", "dns.inflight"}}),
                .Postponed = conf.MetricRegistry.IntGauge({{"sensor", "dns.postponed"}}),
                .Hostnames = conf.MetricRegistry.IntGauge({{"sensor", "dns.hostnames"}}),
                .ResolveStatus = {{
                    conf.MetricRegistry.Rate({{"sensor", "dns.resolve"}, {"status", "OK"}}),
                    conf.MetricRegistry.Rate({{"sensor", "dns.resolve"}, {"status", "UNKNOWN"}}),
                    conf.MetricRegistry.Rate({{"sensor", "dns.resolve"}, {"status", "TIMEOUT"}}),
                    conf.MetricRegistry.Rate({{"sensor", "dns.resolve"}, {"status", "NOT_FOUND"}}),
                }},
                .ResolveTimeMillis = conf.MetricRegistry.HistogramRate(
                        {{"sensor", "dns.resolve.time_millis"}},
                        ExponentialHistogram(16, 2, 1)),
            }
        {
        }

        void Bootstrap(const TActorContext&) {
            Become(&TThis::StateFunc);
            Send(SelfId(), new TEvents::TEvWakeup);
        }

        STFUNC(StateFunc) {
            switch (ev->GetTypeRewrite()) {
                HFunc(TEvStartResolving, OnStartResolving);
                HFunc(TEvStopResolving, OnStopResolving);
                hFunc(TEvResolveDone, OnResolveDone);
                hFunc(TEvResolveFail, OnResolveFail);
                HFunc(TEvPostponed, OnPostponed);
                hFunc(TRackTablesEvents::TDcResult, OnDcResult);
                SFunc(TEvents::TEvWakeup, OnWakeup);
                hFunc(TEvents::TEvPoison, OnPoison);
            }
        }

        STATEFN(Dying) {
            switch (ev->GetTypeRewrite()) {
                sFunc(TEvResolveDone, OnResolveComplete);
                sFunc(TEvResolveFail, OnResolveComplete);
            }
        }

        void OnWakeup(const TActorContext& ctx) {
            ScheduleExpired(ctx);
            Schedule(WakeupInterval_, new TEvents::TEvWakeup);
        }

        void OnPoison(TEvents::TEvPoison::TPtr& ev) {
            if (Metrics_.InFlight->Get() == 0) {
                Send(ev->Sender, new TEvents::TEvPoisonTaken);
                PassAway();
            } else {
                Poisoner_ = ev->Sender;
                Become(&TThis::Dying);
            }
        }

        void OnResolveComplete() {
            if (Metrics_.InFlight->Dec() == 0) {
                Send(Poisoner_, new TEvents::TEvPoisonTaken);
                PassAway();
            }
        }

        void OnStartResolving(const TEvStartResolving::TPtr& evPtr, const TActorContext& ctx) {
            auto&& ev = *evPtr->Get();
            auto& sender = evPtr->Sender;

            // if this url is already scheduled just add a new listener to the list
            // and send resolved url if any
            auto& taskPtr = Tasks_[ev.Hostname];
            if (taskPtr) {
                if (taskPtr->KnownAddress.IsValid()) {
                    Send(sender, new TEvHostResolveOk{taskPtr->Hostname, taskPtr->KnownAddress, taskPtr->KnownDc});
                }
            } else {
                Metrics_.Hostnames->Inc();
                taskPtr = std::make_unique<TTask>(ev.Hostname);
                DoResolve(taskPtr.get(), ctx);
            }

            taskPtr->Listeners.insert(sender);
        }

        void OnStopResolving(const TEvStopResolving::TPtr& evPtr, const TActorContext&) {
            auto&& ev = *evPtr->Get();
            auto it = Tasks_.find(ev.Hostname);
            if (it != Tasks_.end()) {
                it->second->Listeners.erase(evPtr->Sender);
            }
        }

        void OnResolveDone(const TEvResolveDone::TPtr& ev) {
            Metrics_.InFlight->Dec();
            Metrics_.ResolveStatus[0]->Inc();

            auto* task = ev->Get()->Task;
            if (task->KnownAddress != ev->Get()->Address) {
                task->KnownAddress = ev->Get()->Address;
            } else if (task->KnownDc != EDc::UNKNOWN) {
                // address not changed and known DC
                task->LastRequestCompleted = TActivationContext::Now();
                Metrics_.ResolveTimeMillis->Record(task->Duration().MilliSeconds());
                return;
            }

            // resolve DC only if address has changed, or we don't know DC before
            Send(RackTablesActorId_, new TRackTablesEvents::TDcResolve{task->KnownAddress}, 0, reinterpret_cast<ui64>(task));
        }

        void OnResolveFail(const TEvResolveFail::TPtr& ev) {
            Metrics_.InFlight->Dec();
            Metrics_.ResolveStatus[ToUnderlying(ev->Get()->Type) + 1]->Inc();

            auto* task = ev->Get()->Task;
            task->LastRequestCompleted = TActivationContext::Now();
            Metrics_.ResolveTimeMillis->Record(task->Duration().MilliSeconds());

            // notify subscribers only in case hostname was not found
            if (ev->Get()->Type != EErrorType::NotFound) {
                return;
            }

            task->KnownAddress = {};

            for (auto& listenerId: task->Listeners) {
                Send(listenerId, new TEvHostResolveFail{ev->Get()->Type, task->Hostname, ev->Get()->Reason});
            }
        }

        void OnPostponed(const TEvPostponed::TPtr& ev, const TActorContext& ctx) {
            Metrics_.Postponed->Dec();
            DoResolve(ev->Get()->Task, ctx);
        }

        void OnDcResult(const TRackTablesEvents::TDcResult::TPtr& ev) {
            Y_VERIFY(ev->Cookie, "OnDcResult error: cookie is empty");
            auto* task = reinterpret_cast<TTask*>(ev->Cookie);
            task->KnownDc = ev->Get()->Dc;
            task->LastRequestCompleted = TActivationContext::Now();
            Metrics_.ResolveTimeMillis->Record(task->Duration().MilliSeconds());

            for (auto& listenerId: task->Listeners) {
                Send(listenerId, new TEvHostResolveOk{task->Hostname, task->KnownAddress, task->KnownDc});
            }
        }

    private:
        void ScheduleExpired(const TActorContext& ctx) {
            const auto now = ctx.Now();
            /// XXX: maintain a container sorted by time?
            for (auto& [key, task]: Tasks_) {
                if ((now - task->LastRequestStarted) > RefreshInterval_) {
                    // just don't resolve, but leave in cache, since shard may
                    // have been reloaded and we don't want to evict entries completely
                    // in that case
                    if (task->Listeners.empty()) {
                        continue;
                    }

                    DoResolve(task.get(), ctx);
                }
            }
        }

        void DoResolve(TTask* task, const TActorContext& ctx) {
            if (Metrics_.InFlight->Get() >= MAX_IN_FLIGHT) {
                Metrics_.Postponed->Inc();
                Schedule(TDuration::MilliSeconds(RandomNumber(2000u)), new TEvPostponed{task});
                return;
            }

            // we don't really need 100% precision limit here
            Metrics_.InFlight->Inc();

            auto* as = ctx.ExecutorThread.ActorSystem;

            task->LastRequestStarted = TActivationContext::Now();
            DnsClient_->GetAddresses(task->Hostname, true)
                .Subscribe([task, as, selfId = SelfId()] (const TFuture<TIpv6AddressesSet>& f) mutable {
                    try {
                        auto&& val = f.GetValue();
                        if (val.empty()) {
                            as->Send(selfId, new TEvHostResolveFail{
                                EErrorType::NotFound,
                                task->Hostname,
                                TStringBuilder() << "Response for " << task->Hostname << " is empty"
                            });
                            return;
                        }
                        as->Send(selfId, new TEvResolveDone{task, *val.begin()});
                    } catch (const TDnsRequestTimeoutError& e) {
                        as->Send(selfId, new TEvResolveFail{task, EErrorType::Timeout, TString{e.AsStrBuf()}});
                    } catch (const TDnsRecordNotFound& e) {
                        as->Send(selfId, new TEvResolveFail{task, EErrorType::NotFound, TString{e.AsStrBuf()}});
                    } catch (...) {
                        TString msg = CurrentExceptionMessage();
                        MON_WARN_C(*as, DnsResolver, msg);
                        as->Send(selfId, new TEvResolveFail{task, EErrorType::Unknown, msg});
                    }
                });
        }

    private:
        TStringMap<std::unique_ptr<TTask>> Tasks_;
        TDuration WakeupInterval_;
        TDuration RefreshInterval_;
        IDnsClientPtr DnsClient_;
        TActorId RackTablesActorId_;
        TActorId Poisoner_;

        struct {
            TIntGauge* InFlight;
            TIntGauge* Postponed;
            TIntGauge* Hostnames;
            std::array<TRate*, 4> ResolveStatus;
            THistogram* ResolveTimeMillis;
        } Metrics_;
    };
} // namespace

TActorId MakeDnsResolverId() {
    static constexpr TStringBuf ID = "ContDnsRslv";
    return TActorId(0, ID);
}

IActor* CreateDnsResolverActor(TDnsResolverActorConf conf) {
    return new TContinuousResolverActor{std::move(conf)};
}

} // namespace NSolomon::NFetcher
