#include <solomon/services/fetcher/lib/dns/continuous_resolver.h>
#include <solomon/services/fetcher/lib/racktables/racktables_actor.h>
#include <solomon/services/fetcher/testlib/actor_system.h>
#include <solomon/services/fetcher/testlib/http_server.h>

#include <library/cpp/testing/gtest/gtest.h>
#include <library/cpp/threading/future/future.h>

#include <utility>

using namespace NActors;
using namespace NSolomon::NFetcher;
using namespace NSolomon::NTesting;
using namespace NSolomon;
using namespace NThreading;
using namespace testing;

NMonitoring::TMetricRegistry DUMMY_REGISTRY;

struct TMockDnsClient: IDnsClient {
    TFuture<TVector<TSrvRecord>> GetSrvRecords(const TString& address) override {
        if (Err) {
            return MakeErrorFuture<TVector<TSrvRecord>>(Err);
        }

        if (auto* recs = SrvRecords.FindPtr(address)) {
            return MakeFuture(*recs);
        }

        return MakeErrorFuture<TVector<TSrvRecord>>(std::make_exception_ptr(TDnsRecordNotFound()));
    }

    TFuture<TIpv6AddressesSet> GetAddresses(const TString& address, bool) override {
        if (Err) {
            return MakeErrorFuture<TIpv6AddressesSet>(Err);
        }

        if (auto* recs = Addrs.FindPtr(address)) {
            return MakeFuture(*recs);
        }

        return MakeErrorFuture<TIpv6AddressesSet>(std::make_exception_ptr(TDnsRecordNotFound()));
    }

    template <typename T>
    void ThrowOnRequest() {
        Err = std::make_exception_ptr(T{});
    }

    void StopThrowing() {
        Err = {};
    }

    THashMap<TString, TVector<TSrvRecord>> SrvRecords;
    THashMap<TString, TIpv6AddressesSet> Addrs;
    std::exception_ptr Err;
};

TDnsResolverActorConf DEFAULT_CONF {
    .MetricRegistry = DUMMY_REGISTRY,
};

NMonitoring::TMetricRegistry Metrics;

const TString RESPONSE_CONTENT {
R"(2a02:6b8:0:305::/64 [Владимир]
2a02:6b8:0:306::/64 [Владимир]
2a02:6b8:0:408::/64 [Морозов]
2a02:6b8:0:409::/64 [Морозов]
2a02:6b8:0:40a::/64 [Морозов]
2a02:6b8:0:870::/64 [Ивантеевка]
2a02:6b8:0:872::/64 [Ивантеевка]
2a02:6b8:c08:5780::/57 [Сасово-1]
2a02:6b8:c1d:1380::/57 [Владимир-2]
2a02:6b8:b082:15::/64 [офис Мянтсяля]
2a02:6b8:c08:5800::/57 [Сасово-1])"
};

class TContinuousResolverTest: public ::testing::Test {
public:
    void SetUp() override {
        Runtime_ = MakeActorSystem();
        Runtime_->Initialize();
        Runtime_->SetScheduledEventFilter([=](auto&&, auto&&, auto&&, auto&&) { return false; });

        MockClient_ = new TMockDnsClient;
        MockClient_->Addrs.emplace("foo", TIpv6AddressesSet{Get1()});

        Conf_.DnsClient = MockClient_;
        Server_.Reset(new TTestServer);
        TRackTablesActorConf rtConf{
            .RefreshInterval = TDuration::Minutes(1),
            .ConnectTimeout = TDuration::Seconds(5),
            .ReadTimeout = TDuration::Seconds(5),
            .Retries = 0,
            .Url = Server_->Address(),
            .HttpClient = CreateCurlClient({}, Metrics),
            .Registry = DUMMY_REGISTRY,
        };
        Conf_.RackTablesActorId = Runtime_->Register(CreateRackTablesActor(rtConf));
        Server()->AddHandler("/", [] {
            THttpResponse resp;
            resp.SetContent(RESPONSE_CONTENT);
            return resp;
        });

        Edge_ = Runtime_->AllocateEdgeActor();

        InitResolver();
        StartResolving();
    }

protected:
    void InitResolver() {
        Resolver_ = Runtime_->Register(CreateDnsResolverActor(Conf_));
        Runtime_->WaitForBootstrap();
    }

    void StartResolving(TString hostname = "foo") {
        Runtime_->Send(new IEventHandle{Resolver_, Edge_, new TEvStartResolving{std::move(hostname)}});
    }

    void TriggerNextResolve() {
        Runtime_->AdvanceCurrentTime(Conf_.RefreshInterval + TDuration::Seconds(5));
        Runtime_->Send(Resolver_, MakeHolder<TEvents::TEvWakeup>());
    }

    TTestServer* Server() {
        return Server_.Get();
    }

protected:
    TDnsResolverActorConf Conf_{DEFAULT_CONF};
    TAtomicSharedPtr<TMockDnsClient> MockClient_;
    THolder<TTestActorRuntime> Runtime_;
    TActorId Resolver_;
    TActorId Edge_;
    THolder<TTestServer> Server_;
};

TEST_F(TContinuousResolverTest, HostIsResolved) {
    auto ev = Runtime_->GrabEdgeEvent<TEvHostResolveOk>(Edge_)->Release();
    ASSERT_EQ(ev->Hostname, "foo");
    ASSERT_EQ(ev->Address, Get1());
    ASSERT_EQ(ev->Dc, EDc::UNKNOWN);
}

TEST_F(TContinuousResolverTest, NextResolveIsScheduled) {
    auto scheduledEvs = Runtime_->CaptureScheduledEvents();
    ASSERT_THAT(scheduledEvs.begin()->Event->GetTypeRewrite(), Eq(TEvents::TEvWakeup::EventType));
    Runtime_->AdvanceCurrentTime(Conf_.RefreshInterval);
}

TEST_F(TContinuousResolverTest, SubscriberIsNotifiedOnAddressChange) {
    Runtime_->GrabEdgeEvent<TEvHostResolveOk>(Edge_);
    bool ok;
    // address with a known DC (VLA in this test)
    auto newAddr = TIpv6Address::FromString("2a02:6b8:0:305:0000:0000:0000:000", ok);

    MockClient_->Addrs["foo"] = {newAddr};
    TriggerNextResolve();
    {
        auto ev = Runtime_->GrabEdgeEvent<TEvHostResolveOk>(Edge_)->Release();

        ASSERT_EQ(ev->Hostname, "foo");
        ASSERT_EQ(ev->Address, newAddr);
    }

    MockClient_->Addrs.erase("foo");
    TriggerNextResolve();
    {
        auto ev = Runtime_->GrabEdgeEvent<TEvHostResolveFail>(Edge_)->Release();

        ASSERT_EQ(ev->Hostname, "foo");
    }

    // the same address as before
    MockClient_->Addrs["foo"] = {newAddr};
    TriggerNextResolve();
    {
        auto ev = Runtime_->GrabEdgeEvent<TEvHostResolveOk>(Edge_)->Release();

        ASSERT_EQ(ev->Hostname, "foo");
        ASSERT_EQ(ev->Address, newAddr);
    }
}

TEST_F(TContinuousResolverTest, RecordGetsRestoredAfterNotFound) {
    Runtime_->GrabEdgeEvent<TEvHostResolveOk>(Edge_);
    MockClient_->ThrowOnRequest<TDnsRecordNotFound>();

    {
        TriggerNextResolve();
        auto ev = Runtime_->GrabEdgeEvent<TEvHostResolveFail>(Edge_)->Release();
        ASSERT_EQ(ev->Type, EErrorType::NotFound);
    }

    bool ok;
    MockClient_->StopThrowing();
    MockClient_->Addrs["foo"] = TIpv6AddressesSet{TIpv6Address::FromString("2a02:6b8:c03:35d:0:4c7d:b2a9:67d2", ok)};
    TriggerNextResolve();

    auto ev = Runtime_->GrabEdgeEvent<TEvHostResolveOk>(Edge_)->Release();
    ASSERT_TRUE(ev);
}

TEST_F(TContinuousResolverTest, RecordIsNotChangedOnUnknown) {
    Runtime_->GrabEdgeEvent<TEvHostResolveOk>(Edge_);
    MockClient_->ThrowOnRequest<TDnsClientInternalError>();

    {
        TriggerNextResolve();
        auto ev = Runtime_->GrabEdgeEvent<TEvHostResolveOk>(Edge_, TDuration::Seconds(1));
        ASSERT_FALSE(ev);
    }

    bool ok;
    MockClient_->StopThrowing();
    MockClient_->Addrs["foo"] = TIpv6AddressesSet{TIpv6Address::FromString("2a02:6b8:c03:35d:0:4c7d:b2a9:67d2", ok)};
    TriggerNextResolve();

    auto ev = Runtime_->GrabEdgeEvent<TEvHostResolveOk>(Edge_)->Release();
    ASSERT_TRUE(ev);
}
