#include "download.h"

#include <solomon/libs/cpp/http/client/http.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/event_local.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/actors/core/log.h>
#include <library/cpp/http/misc/http_headers.h>

#include <util/datetime/cputimer.h>

#include <utility>

namespace NSolomon::NFetcher {
    namespace {
        using namespace NActors;
        const TString VALUE_USER_AGENT = TStringBuilder() << "Solomon-Fetcher/" << GetProgramSvnRevision();

        class TFetcherActor: public TActorBootstrapped<TFetcherActor>, private TPrivateEvents {
            enum {
                EvResponse = SpaceBegin,
                End,
            };
            static_assert(End < SpaceEnd, "too many event types");

            struct TEvResponse: TEventLocal<TEvResponse, EvResponse> {
                TEvResponse(IHttpClient::TResult result)
                    : Result{std::move(result)}
                {
                }

                IHttpClient::TResult Result;
            };

        public:
            explicit TFetcherActor(const TFetcherActorConfig& conf, TDownloadRequest&& req, TActorId receiver)
                : Client_{conf.Client}
                , Timeout_{conf.Timeout}
                , ConnectTimeout_{conf.ConnectTimeout}
                , Request_{std::move(req)}
                , Receiver_{receiver}
            {
            }

            void Bootstrap(const TActorContext& ctx) {
                Url_ = std::move(Request_.Url);
                auto& tsArgs = Request_.TsArgs;

                auto url = Url_ + tsArgs;
                auto& body = Request_.Body;

                auto headers = Headers();
                for (auto&& [k, v]: Request_.Headers) {
                    headers->Add(k, v);
                }

                headers->Add(TStringBuf("User-Agent"), VALUE_USER_AGENT);

                TRequestOpts opts;
                opts.ConnectTimeout = ConnectTimeout_;
                opts.ReadTimeout = Timeout_;
                opts.Retries = 0;

                const auto responseLimit = Request_.ResponseSizeLimitBytes;
                if (responseLimit != 0) {
                    opts.ResponseLimitBytes = responseLimit;
                }

                const auto selfId = SelfId();

                Start_ = Request_.DownloadInstant
                    ? Request_.DownloadInstant
                    : TInstant::Now();

                const auto* actorSystem = ctx.ExecutorThread.ActorSystem;
                auto cb = [=] (auto result) {
                    actorSystem->Send(selfId, new TEvResponse{std::move(result)});
                };

                Timer_.Reset();

                const auto isPostRequest = !body.empty() || Request_.IsPost;

                if (isPostRequest) {
                    Client_->Request(
                        Post(std::move(url), std::move(body), std::move(headers)),
                        std::move(cb),
                        std::move(opts)
                    );
                } else {
                    Client_->Request(
                        Get(std::move(url), std::move(headers)),
                        std::move(cb),
                        std::move(opts)
                    );
                }

                Become(&TThis::StateFunc);
            }

            STFUNC(StateFunc) {
                Y_UNUSED(ctx);
                switch (ev->GetTypeRewrite()) {
                    cFunc(TEvents::TSystem::PoisonPill, PassAway);
                    hFunc(TEvResponse, OnResponse);
                };
            }

            void OnResponse(const TEvResponse::TPtr& response) {
                auto& result = response->Get()->Result;
                auto ev = MakeHolder<TEvDownloadCompleted>();
                ev->Duration = Timer_.Get();
                ev->DownloadStart = Start_;
                ev->Result = std::move(result);

                Send(Receiver_, ev.Release());
                PassAway();
            }

        private:
            TString Url_;
            IHttpClientPtr Client_;
            TDuration Timeout_;
            TDuration ConnectTimeout_;
            TInstant Start_;
            TSimpleTimer Timer_;
            TDownloadRequest Request_;
            TActorId Receiver_;
        };
    } // namespace

    // NOLINTNEXTLINE(performance-unnecessary-value-param): false positive
    IActor* CreateHttpDownloadActor(TFetcherActorConfig conf, TDownloadRequest&& req, TActorId receiver) {
        return new TFetcherActor{std::move(conf), std::move(req), receiver};
    }

} // namespace NSolomon::NFetcher
