#include <solomon/services/fetcher/lib/download/download.h>
#include <solomon/services/fetcher/testlib/actor_system.h>
#include <solomon/services/fetcher/testlib/http_server.h>

#include <solomon/libs/cpp/http/client/http.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/http/fetch/exthttpcodes.h>
#include <library/cpp/http/misc/http_headers.h>
#include <library/cpp/http/server/response.h>
#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/testing/gtest/gtest.h>

using namespace NActors;
using namespace NSolomon;
using namespace NSolomon::NFetcher;
using namespace NSolomon::NTesting;

class TFetcherActorTest: public ::testing::Test {
public:
    void SetUp() override {
        Srv_ = MakeHolder<TTestServer>();
        Srv_->AddHandler("/", [] {
            THttpResponse resp{HTTP_OK};
            resp.SetContent("hello, world!");
            return resp;
        });

        ActorSystem_ = MakeActorSystem();
        ActorSystem_->Initialize();

        HttpClient_ = CreateCurlClient({}, Registry_);
    }

    void TearDown() override {
        HttpClient_.Reset();
        Srv_->Stop();
        ActorSystem_.Reset();
    }

    void MakeRequest(TDownloadRequest&& r, TActorId rcv) {
        ActorSystem_->Register(CreateHttpDownloadActor(
            TFetcherActorConfig{HttpClient_, TDuration::Seconds(2), TDuration::Seconds(1)},
            std::move(r),
            rcv
        ));
    }

protected:
    NMonitoring::TMetricRegistry Registry_;
    THolder<TTestServer> Srv_;
    THolder<TFetcherActorRuntime> ActorSystem_;
    IHttpClientPtr HttpClient_;
    TActorId TestActor_;
};

TEST_F(TFetcherActorTest, FetchSimpleData) {
    auto actor = ActorSystem_->AllocateEdgeActor();
    MakeRequest(TDownloadRequest{Srv_->Address()}, actor);

    auto ev = ActorSystem_->GrabEdgeEvent<TEvDownloadCompleted>(actor);
    auto& result = ev->Get()->Result;

    ASSERT_TRUE(result.Success());
    ASSERT_EQ(result.Value()->Code(), HTTP_OK);
    ASSERT_EQ(result.Value()->Data(), "hello, world!");
}

TEST_F(TFetcherActorTest, FetchTimeout) {
    auto handler = [] {
        THttpResponse resp;
        Sleep(TDuration::Seconds(3));
        resp.SetHttpCode(HTTP_INTERNAL_SERVER_ERROR);
        resp.SetContent("hello, world!");
        return resp;
    };

    Srv_->AddHandler("/", handler);
    auto actor = ActorSystem_->AllocateEdgeActor();
    TDownloadRequest r{Srv_->Address(), {}, {}, 0};
    MakeRequest(std::move(r), actor);

    auto ev = ActorSystem_->GrabEdgeEvent<TEvDownloadCompleted>(actor);
    auto& result = ev->Get()->Result;

    ASSERT_TRUE(!result.Success());
    ASSERT_EQ(result.Error().Type(), TRequestError::EType::ReadTimeout);
}

TEST_F(TFetcherActorTest, FetchDnsError) {
    auto actor = ActorSystem_->AllocateEdgeActor();
    TDownloadRequest r{"http://someaddress.which.does.not.exist", {}, {}, 0};
    MakeRequest(std::move(r), actor);

    auto ev = ActorSystem_->GrabEdgeEvent<TEvDownloadCompleted>(actor);
    auto& result = ev->Get()->Result;

    ASSERT_TRUE(!result.Success());
    ASSERT_EQ(result.Error().Type(), TRequestError::EType::DnsFailure);
}

TEST_F(TFetcherActorTest, FetchTooLarge) {
    auto actor = ActorSystem_->AllocateEdgeActor();
    TDownloadRequest r{Srv_->Address(), {}, {}, 1};
    MakeRequest(std::move(r), actor);

    auto ev = ActorSystem_->GrabEdgeEvent<TEvDownloadCompleted>(actor);
    auto& result = ev->Get()->Result;

    ASSERT_TRUE(!result.Success());
    ASSERT_EQ(result.Error().Type(), TRequestError::EType::ResponseTooLarge);
}
