#pragma once

#include <solomon/services/fetcher/lib/cluster/cluster.h>
#include <solomon/services/fetcher/lib/host_groups/host_resolver_iface.h>

#include <solomon/services/fetcher/lib/id.h>

#include <solomon/libs/cpp/conf_db/model/agent_config.h>
#include <solomon/libs/cpp/conf_db/model/cluster_config.h>
#include <solomon/libs/cpp/conf_db/model/provider_config.h>
#include <solomon/libs/cpp/conf_db/model/service_config.h>
#include <solomon/libs/cpp/conf_db/model/shard_config.h>
#include <solomon/libs/cpp/error_or/error_or.h>

#include <library/cpp/threading/future/future.h>

#include <util/generic/ptr.h>
#include <util/system/hostname.h>

#include <optional>

namespace NSolomon::NFetcher {
    using TShardConfigPtr = TAtomicSharedPtr<const NDb::NModel::TShardConfig>;
    using TServiceConfigPtr = TAtomicSharedPtr<const NDb::NModel::TServiceConfig>;
    using TClusterConfigPtr = TAtomicSharedPtr<const NDb::NModel::TClusterConfig>;
    using TProviderConfigPtr = TAtomicSharedPtr<const NDb::NModel::TProviderConfig>;

    using TMaybeTvmId = std::optional<ui32>;

    enum class EFetcherShardType {
        Simple = 0,
        Agent = 1,
        YasmAgent = 2,
    };

    class IHostResolverFactory;

    class IFetcherCluster: public TThrRefBase {
    public:
        virtual const TClusterId& Id() const = 0;
        virtual TVector<TErrorOr<IHostGroupResolverPtr, TGenericError>> CreateResolvers(IHostResolverFactory&) const = 0;
        virtual const TString& Name() const = 0;

        virtual EFetcherShardType Type() const = 0;
        virtual bool Equals(const IFetcherCluster&) const = 0;
    };

    using IFetcherClusterPtr = TIntrusiveConstPtr<IFetcherCluster>;

    IFetcherClusterPtr CreateCluster(const NDb::NModel::TClusterConfig& clusterConfig);

    class IFetcherShard: public TThrRefBase {
    public:
        virtual const TShardId& Id() const = 0;
        virtual TStringBuf ProjectId() const = 0;
        virtual TStringBuf ServiceName() const = 0;
        // TODO: add ClusterName() method

        virtual NDb::NModel::EPullProtocol Protocol() const = 0;
        virtual ui16 Port() const = 0;
        virtual TMaybeTvmId TvmId() const = 0;
        virtual TStringBuf UrlPath() const = 0;
        virtual TDuration FetchInterval() const = 0 ;
        virtual i32 GridSec() const = 0;

        virtual bool IsAddTsArgs() const = 0;
        virtual bool IsUseFqdn() const = 0;
        virtual bool IsPull() const = 0;
        virtual bool IsValid() const = 0;

        virtual bool IsEqual(const IFetcherShard& other) const = 0;

        virtual ui32 MaxResponseSizeBytes() const = 0;
        virtual ui32 MaxMetricsPerUrl() const = 0;

        virtual const TClusterNode& Location() const = 0;
        virtual TIntrusiveConstPtr<IFetcherCluster> Cluster() const = 0;

        virtual EFetcherShardType Type() const = 0;
        virtual TString ToString() const = 0;

        virtual bool IsLocal() const {
            static const auto fqdn = FQDNHostName();
            return Location().Fqdn == fqdn;
        }

        virtual void DumpConfig(IOutputStream* out) const = 0;
    };

    class TFetcherShard final: public IFetcherShard {
    public:
        explicit TFetcherShard(TIntrusiveConstPtr<IFetcherShard> impl)
            : Impl_{std::move(impl)}
        {
        }

        const TShardId& Id() const override {
            return Impl_->Id();
        }

        TStringBuf ProjectId() const override {
            return Impl_->ProjectId();
        }

        TStringBuf ServiceName() const override {
            return Impl_->ServiceName();
        }

        NDb::NModel::EPullProtocol Protocol() const override {
            return Impl_->Protocol();
        }

        ui16 Port() const override {
            return Impl_->Port();
        }

        TMaybeTvmId TvmId() const override {
            return Impl_->TvmId();
        }

        TStringBuf UrlPath() const override {
            return Impl_->UrlPath();
        }

        TDuration FetchInterval() const override {
            return Impl_->FetchInterval();
        }

        i32 GridSec() const override {
            return Impl_->GridSec();
        }

        bool IsAddTsArgs() const override {
            return Impl_->IsAddTsArgs();
        }

        bool IsUseFqdn() const override {
            return Impl_->IsUseFqdn();
        }

        bool IsPull() const override {
            return Impl_->IsPull();
        }

        bool IsValid() const override {
            return Impl_->IsValid();
        }

        bool IsEqual(const IFetcherShard& other) const override {
            if (Type() != other.Type()) {
                return false;
            }

            return other.IsEqual(*Impl_);
        }

        ui32 MaxResponseSizeBytes() const override {
            return Impl_->MaxResponseSizeBytes();
        }

        ui32 MaxMetricsPerUrl() const override {
            return Impl_->MaxMetricsPerUrl();
        }

        const TClusterNode& Location() const override {
            return Impl_->Location();
        }

        TIntrusiveConstPtr<IFetcherCluster> Cluster() const override {
            return Impl_->Cluster();
        }

        EFetcherShardType Type() const override {
            return Impl_->Type();
        }

        TString ToString() const override {
            return Impl_->ToString();
        }

        bool IsLocal() const override {
            return Impl_->IsLocal();
        }

        void DumpConfig(IOutputStream* out) const override {
            Impl_->DumpConfig(out);
        }

    private:
        TIntrusiveConstPtr<IFetcherShard> Impl_;
    };

    TFetcherShard CreateSimpleShard(
        TShardConfigPtr shard,
        TClusterConfigPtr cluster,
        TServiceConfigPtr service,
        const TClusterNode& location = TClusterNode{}
    );

    TFetcherShard CreateAgentShard(
        TStringBuf providerId,
        TDuration pullInterval,
        TVector<NDb::NModel::TAgentConfig> agents
    );

    TFetcherShard CreateYasmAgentShard();
    TFetcherShard CreateYasmAgentShard(TVector<TString> walleTags);
    TIntrusiveConstPtr<IFetcherCluster> CreateYasmAgentCluster();

    inline IOutputStream& operator<<(IOutputStream& os, const NSolomon::NFetcher::TFetcherShard& shard) {
        os << shard.ToString();
        return os;
    }

    inline TString MakeAgentShardId(const TString& providerId) {
        return providerId + "_provider_" + providerId + "_agent";
    }
} // namespace NSolomon::NFetcher

template <>
struct THash<NSolomon::NFetcher::TFetcherShard> {
    ui32 operator()(const NSolomon::NFetcher::IFetcherShard& shard) const {
        return shard.Id().Hash();
    }
};
