#include <solomon/services/fetcher/lib/fetcher_shard.h>
#include <solomon/services/fetcher/lib/host_groups/host_and_labels.h>
#include <solomon/services/fetcher/lib/host_groups/agents.h>

#include <solomon/libs/cpp/conf_db/model/agent_config.h>

#include <util/generic/size_literals.h>
#include <util/string/builder.h>

#include <optional>

namespace NSolomon::NFetcher {
namespace {
    static constexpr auto AGENT_SERVICE = "agent";
    constexpr TDuration FALLBACK_FETCH_INTERVAL = TDuration::Seconds(10); // minimal supported value

    THostAndLabels CreateFromAgentConfig(const NDb::NModel::TAgentConfig& conf) {
        THostAndLabels hl{
            conf.Hostname,
            conf.DataPort,
            Nothing(),
        };

        return hl;
    }

    class TAgentCluster: public IFetcherCluster {
        static const TString GLOBAL_CLUSTER;

    public:
        TAgentCluster(TString projectId, const TString& clusterName, TVector<NDb::NModel::TAgentConfig> agents)
            : Agents_{std::move(agents)}
            , ClusterName_{clusterName.empty() ? GLOBAL_CLUSTER : clusterName}
            , Id_{ClusterName_, std::move(projectId)}
        {
        }

        TAgentCluster(TClusterId id, TVector<NDb::NModel::TAgentConfig> agents)
            : Agents_{std::move(agents)}
            , Id_{std::move(id)}
        {
        }

        const TClusterId& Id() const override {
            return Id_;
        }

        const TString& Name() const override {
            return Id_.ClusterId();
        }

        TVector<TErrorOr<IHostGroupResolverPtr, TGenericError>> CreateResolvers(IHostResolverFactory& factory) const override {
            TVector<THostAndLabels> hosts;
            for (auto&& agent: Agents_) {
                hosts.push_back(CreateFromAgentConfig(agent));
            }

            TVector<TErrorOr<IHostGroupResolverPtr, TGenericError>> result;
            result.push_back(factory.CreateAgentGroupResolver(Id_.ToString(), std::move(hosts)));

            return result;
        }

        EFetcherShardType Type() const override {
            return EFetcherShardType::Agent;
        }

        bool Equals(const IFetcherCluster& other) const override {
            if (Type() != other.Type()) {
                return false;
            }

            return static_cast<const TAgentCluster&>(other).Agents_ == Agents_;
        }

        const TVector<NDb::NModel::TAgentConfig>& Agents() const {
            return Agents_;
        }

    private:
        TVector<NDb::NModel::TAgentConfig> Agents_;
        TString ClusterName_;
        TClusterId Id_;
    };

    const TString TAgentCluster::GLOBAL_CLUSTER{"global"};

    TString MakeId(const TString& projectId, const IFetcherCluster& cluster) {
        return TStringBuilder() << projectId << '_' << cluster.Name() << '_' << AGENT_SERVICE;
    }

    // dirty fix for SOLOMON-7415
    TDuration GetMaxPullInterval(const TVector<NDb::NModel::TAgentConfig>& agents) {
        TDuration pullInterval;

        for (const auto& agent: agents) {
            pullInterval = Max(pullInterval, agent.PullInterval);
        }

        return pullInterval;
    }

    class TAgentShard: public IFetcherShard {
        static constexpr TStringBuf URL_PATH = "/storage/readAll";

    public:
        TAgentShard(TStringBuf providerId, TDuration pullInterval, TVector<NDb::NModel::TAgentConfig> agents)
            : ProviderId_{providerId}
            , Cluster_{new TAgentCluster{TClusterId{TString{"provider_"} + ProviderId_, ProviderId_}, std::move(agents)}}
            , Id_{MakeId(ProviderId_, *Cluster_), 0}
            , FetchInterval_{pullInterval}
        {
        }

        const TShardId& Id() const override {
            return Id_;
        }

        TStringBuf ProjectId() const override {
            return ProviderId_;
        }

        TStringBuf ServiceName() const override {
            return {};
        }

        NDb::NModel::EPullProtocol Protocol() const override {
            return NDb::NModel::EPullProtocol::HTTP;
        }

        ui16 Port() const override {
            return 0;
        }

        TMaybeTvmId TvmId() const override {
            return {};
        }

        TStringBuf UrlPath() const override {
            return URL_PATH;
        }

        TDuration FetchInterval() const override {
            return FetchInterval_;
        }

        i32 GridSec() const override {
            return NDb::NModel::TServiceConfig::GRID_ABSENT;
        }

        bool IsAddTsArgs() const override {
            return false;
        }

        bool IsUseFqdn() const override {
            return true;
        }

        bool IsPull() const override {
            return true;
        }

        bool IsValid() const override {
            return true;
        }

        bool IsEqual(const IFetcherShard& other) const override {
            return Cluster_->Equals(*other.Cluster()) && FetchInterval_ == other.FetchInterval();
        }

        ui32 MaxResponseSizeBytes() const override {
            return 100_MB;
        }

        ui32 MaxMetricsPerUrl() const override {
            return 500'000;
        }

        const TClusterNode& Location() const override {
            return Location_;
        }

        TIntrusiveConstPtr<IFetcherCluster> Cluster() const override {
            return Cluster_;
        }

        EFetcherShardType Type() const override {
            return EFetcherShardType::Agent;
        }

        TString ToString() const override {
            return {};
        }

        bool IsLocal() const override {
            return true;
        }

        void DumpConfig(IOutputStream* out) const override {
            *out << "Location: " << Location_ << '\n';
            *out << "ProviderId: " << ProviderId_ << '\n';
            *out << "ClusterId: " << ClusterId_ << '\n';
            *out << "Id: " << Id_.ToString() << '\n';
            *out << "FetchInterval: " << FetchInterval_ << '\n';
        }

    private:
        TClusterNode Location_;
        TString ProviderId_;
        TString ClusterId_;
        TIntrusivePtr<IFetcherCluster> Cluster_;
        // NOTE: beware that this is not ShardId and cannot be used as such
        TShardId Id_;
        TDuration FetchInterval_;
    };
} // namespace
    TFetcherShard CreateAgentShard(
        TStringBuf providerId,
        TDuration pullInterval,
        TVector<NDb::NModel::TAgentConfig> agents)
    {
        auto interval = pullInterval ? pullInterval : Max(GetMaxPullInterval(agents), FALLBACK_FETCH_INTERVAL);

        return TFetcherShard{
            new TAgentShard{providerId, interval, std::move(agents)}
        };
    }

} // namespace NSolomon::NFetcher
