#include "simple_shard.h"

#include <solomon/services/fetcher/lib/host_groups/config_parser.h>
#include <solomon/services/fetcher/lib/host_groups/factory.h>

#include <solomon/libs/cpp/conf_db/model/cluster_config.h>
#include <solomon/libs/cpp/conf_db/model/service_config.h>
#include <solomon/libs/cpp/conf_db/model/shard_config.h>

#include <util/string/builder.h>

#include <utility>

namespace NSolomon::NFetcher {
namespace {

using namespace NDb::NModel;

static constexpr ui16 DEFAULT_HTTP_PORT = 80;
static constexpr ui16 DEFAULT_HTTPS_PORT = 443;

    class TFetcherCluster: public IFetcherCluster {
    public:
        TFetcherCluster(TClusterConfigPtr cluster)
            : Cluster_{std::move(cluster)}
            , Id_{Cluster_->Id, Cluster_->ProjectId}
        {
        }

        const TClusterId& Id() const override {
            return Id_;
        }

        const TString& Name() const override {
            return Cluster_->Name;
        }

        EFetcherShardType Type() const override {
            return EFetcherShardType::Simple;
        }

        TVector<TErrorOr<IHostGroupResolverPtr, TGenericError>> CreateResolvers(IHostResolverFactory& factory) const override {
            using TResult = TErrorOr<IHostGroupResolverPtr, TGenericError>;
            TVector<TResult> result;

            auto forEachConf = [&] (auto&& parse, TString field, auto&& fn) {
                decltype(parse(field)) confs;

                try {
                    confs = parse(field);
                } catch (...) {
                    result.push_back(TResult::FromError(
                        TStringBuilder() << "Failed to parse " << field << ": " << CurrentExceptionMessage()
                    ));
                    return;
                }

                if (confs.empty()) {
                    return;
                }

                for (auto&& conf: confs) {
                    try {
                        IHostGroupResolverPtr resolver = fn(std::move(conf));
                        result.push_back(TResult::FromValue(std::move(resolver)));
                    } catch (...) {
                        result.push_back(TResult::FromError(
                            TStringBuilder() << "Failed to create resolver from " << field << ": " << CurrentExceptionMessage()
                        ));
                    }
                }
            };

            auto&& cluster = *Cluster_;
            if (cluster.ConductorGroups) {
                forEachConf(ParseConductorGroups, cluster.ConductorGroups, [&] (TConductorConfig&& clusterConf) {
                    return factory.CreateConductorGroupResolver(std::move(clusterConf));
                });
            }

            if (cluster.ConductorTags) {
                forEachConf(ParseConductorTags, cluster.ConductorTags, [&] (TConductorConfig&& clusterConf) {
                    return factory.CreateConductorTagResolver(std::move(clusterConf));
                });
            }

            if (cluster.HostUrls) {
                forEachConf(ParseHostUrlConfig, cluster.HostUrls, [&] (THostListUrlConfig&& clusterConf) {
                    return factory.CreateHostUrlResolver(std::move(clusterConf));
                });
            }

            if (cluster.Hosts) {
                forEachConf(ParseHostPatternConfig, cluster.Hosts, [&] (THostPatternConfig&& clusterConf) {
                    return factory.CreateHostPatternResolver(std::move(clusterConf));
                });
            }

            if (cluster.QloudGroups) {
                forEachConf(ParseQloudConfig, cluster.QloudGroups, [&] (TQloudConfig&& clusterConf) {
                    return factory.CreateQloudResolver(std::move(clusterConf));
                });
            }

            if (cluster.Networks) {
                forEachConf(ParseNetworkConfig, cluster.Networks, [&] (TNetworkConfig&& networkConf) {
                    return factory.CreateNetworkResolver(std::move(networkConf));
                });
            }

            if (cluster.NannyGroups) {
                forEachConf(ParseNannyConfig, cluster.NannyGroups, [&] (TNannyConfig&& nannyConf) {
                    return factory.CreateNannyResolver(std::move(nannyConf));
                });
            }

            if (cluster.YpClusters) {
                forEachConf(ParseYpConfig, cluster.YpClusters, [&] (TYpConfig&& ypConf) {
                    return factory.CreateYpResolver(std::move(ypConf));
                });
            }

            if (cluster.InstanceGroups) {
                forEachConf(ParseInstanceGroupConfig, cluster.InstanceGroups, [&] (TInstanceGroupConfig&& igConf) {
                    return factory.CreateInstanceGroupResolver(std::move(igConf));
                });
            }

            if (cluster.CloudDns) {
                forEachConf(ParseCloudDnsConfig, cluster.CloudDns, [&] (TCloudDnsConfig&& conf) {
                    return factory.CreateCloudDnsResolver(cluster.ProjectId, std::move(conf));
                });
            }

            return result;
        }

        bool Equals(const IFetcherCluster& other) const override {
            if (Type() != other.Type()) {
                return false;
            }

            return !NDb::NModel::AnyHostGroupChanged(
                *(static_cast<const TFetcherCluster&>(other).Cluster_),
                *Cluster_
            );
        }

    private:
        TClusterConfigPtr Cluster_;
        TClusterId Id_;
    };
} // namespace

    TSimpleShard::TSimpleShard(TShardConfigPtr shard, TClusterConfigPtr cluster, TServiceConfigPtr service, const TClusterNode& location)
        : Shard_{std::move(shard)}
        , Cluster_{std::move(cluster)}
        , Service_{std::move(service)}
        , FetcherCluster_{new TFetcherCluster{Cluster_}}
        , Location_{location}
        , ShardId_{Shard_->Id, Shard_->NumId}
    {
    }

    NDb::NModel::EPullProtocol TSimpleShard::Protocol() const {
        auto protocol = GetPullSetting<EPullProtocol>([](const TPullSettings* pull) {
            return (pull->Protocol != EPullProtocol::UNKNOWN) ? std::make_optional(pull->Protocol) : std::nullopt;
        });
        return protocol.value_or(NDb::NModel::EPullProtocol::HTTP);
    }

    ui16 TSimpleShard::Port() const {
        auto port = GetPullSetting<ui16>([](const TPullSettings* pull) {
            return pull->Port ? std::make_optional(pull->Port) : std::nullopt;
        });

        if (port) {
            return *port;
        }

        switch (Protocol()) {
            case EPullProtocol::HTTP: return DEFAULT_HTTP_PORT;
            case EPullProtocol::HTTPS: return DEFAULT_HTTPS_PORT;
            case EPullProtocol::UNKNOWN: return DEFAULT_HTTP_PORT;
        }
    }

    const TShardId& TSimpleShard::Id() const {
        return ShardId_;
    }

    TStringBuf TSimpleShard::ServiceName() const {
        return Service_->Name;
    }

    TStringBuf TSimpleShard::ProjectId() const {
        return Shard_->ProjectId;
    }

    TMaybeTvmId TSimpleShard::TvmId() const {
        return GetPullSetting<ui32>([](const TPullSettings* pull) {
            return pull->TvmDestId ? std::make_optional(pull->TvmDestId) : std::nullopt;
        });
    }

    TStringBuf TSimpleShard::UrlPath() const {
        auto path = GetPullSetting<TStringBuf>([](const TPullSettings* pull) {
            return pull->Path.empty() ? std::nullopt : std::make_optional(TStringBuf{pull->Path});
        });
        return path.value_or(TStringBuf{});
    }

    TDuration TSimpleShard::FetchInterval() const {
        return Service_->Interval;
    }

    i32 TSimpleShard::GridSec() const {
        return Service_->GridSec;
    }

    bool TSimpleShard::IsAddTsArgs() const {
        auto addTsArgs = GetPullSetting<bool>([](const TPullSettings* pull) {
            return pull->AddTsArgs ? std::make_optional(true) : std::nullopt;
        });
        return addTsArgs.value_or(false);
    }

    bool TSimpleShard::IsUseFqdn() const {
        auto hostLabelPolicy = GetPullSetting<EHostLabelPolicy>([](const TPullSettings* pull) {
            return (pull->HostLabelPolicy != EHostLabelPolicy::UNKNOWN)
                ? std::make_optional(pull->HostLabelPolicy)
                : std::nullopt;
        });
        return hostLabelPolicy.has_value() && *hostLabelPolicy == EHostLabelPolicy::FULL_HOSTNAME;
    }

    bool TSimpleShard::IsPull() const {
        return !UrlPath().empty();
    }

    bool TSimpleShard::IsValid() const {
        return Shard_ && Cluster_ && Service_;
    }

    ui32 TSimpleShard::MaxResponseSizeBytes() const {
        return Shard_->MaxResponseSizeBytes;
    }

    ui32 TSimpleShard::MaxMetricsPerUrl() const {
        return Shard_->MaxMetricsPerUrl;
    }

    const TClusterNode& TSimpleShard::Location() const {
        return Location_;
    }

    TIntrusiveConstPtr<IFetcherCluster> TSimpleShard::Cluster() const {
        return FetcherCluster_;
    }

    // this version does compare locations of the shards
    bool TSimpleShard::IsEqual(const IFetcherShard& other) const {
        const auto& o = static_cast<const TSimpleShard&>(other);
        auto eq = [] (auto&& lhs, auto&& rhs) {
            return (!lhs && !rhs)
                || (*lhs == *rhs);
        };

        return eq(o.Shard_, Shard_)
            && eq(o.Cluster_, Cluster_)
            && eq(o.Service_, Service_);
    }

    EFetcherShardType TSimpleShard::Type() const {
        return EFetcherShardType::Simple;
    }

    TString TSimpleShard::ToString() const {
        return TStringBuilder() << *Shard_ << '\n' << *Cluster_ << '\n' << *Service_;
    }

    void TSimpleShard::DumpConfig(IOutputStream* out) const {
        *out << "Shard: {" << *Shard_ << "}\n\n";
        *out << "Cluster: {" << *Cluster_ << "}\n\n";
        *out << "Service: {" << *Service_ << '}';
    }

    TFetcherShard CreateSimpleShard(
        TShardConfigPtr shard,
        TClusterConfigPtr cluster,
        TServiceConfigPtr service,
        const TClusterNode& location)
    {
        return TFetcherShard{new TSimpleShard{std::move(shard), std::move(cluster), std::move(service), location}};
    }

    IFetcherClusterPtr CreateCluster(const TClusterConfig& clusterConfig) {
        return new TFetcherCluster{MakeAtomicShared<TClusterConfig>(clusterConfig)};
    }
} // namespace NSolomon::NFetcher
