#include <solomon/services/fetcher/lib/fetcher_shard.h>
#include <solomon/services/fetcher/lib/host_groups/agents.h>
#include <solomon/services/fetcher/lib/host_groups/host_resolver.h>

#include <solomon/libs/cpp/yasm/constants/interval.h>

#include <util/generic/size_literals.h>

namespace NSolomon::NFetcher {
namespace {
    constexpr ui16 YASM_PORT = 11003;
    constexpr ui32 METRIC_LIMIT = 500'000;
    constexpr ui32 RESPONSE_SIZE_LIMIT = 100_MB;

    const TString CLUSTER_NAME{"global"};

    class TYasmCluster: public IFetcherCluster {
    public:
        TYasmCluster() = default;

        explicit TYasmCluster(TVector<TString> walleTags)
            : WalleTags_{std::move(walleTags)}
        {
        }

        const TClusterId& Id() const override{
            return ClusterId_;
        }

        TVector<TErrorOr<IHostGroupResolverPtr, TGenericError>> CreateResolvers(IHostResolverFactory& factory) const override {
            TVector<TErrorOr<IHostGroupResolverPtr, TGenericError>> result;
            result.emplace_back(factory.CreateYasmAgentGroupResolver(WalleTags_, ConductorTags_));
            return result;
        }

        const TString& Name() const override {
            return CLUSTER_NAME;
        }

        EFetcherShardType Type() const override {
            return EFetcherShardType::YasmAgent;
        }

        bool Equals(const IFetcherCluster& other) const override {
            if (other.Type() != Type()) {
                return false;
            }

            const auto& o = static_cast<const TYasmCluster&>(other);
            auto&& end = std::make_pair(WalleTags_.end(), o.WalleTags_.end());
            return Mismatch(WalleTags_.begin(), WalleTags_.end(),
                            o.WalleTags_.begin(), o.WalleTags_.end()) == end;
        }

    private:
        const TVector<TString> WalleTags_{"yasm_monitored", "qloud_yasm_monitored"};
        const TVector<TString> ConductorTags_{"yasm_monitored"};

        const TClusterId ClusterId_{"yasm_agent", "solomon"};
    };

    class TYasmShard: public IFetcherShard {
    public:
        TYasmShard()
            : Cluster_{new TYasmCluster}
        {
        }

        TYasmShard(TVector<TString> walleTags)
            : Cluster_{new TYasmCluster{std::move(walleTags)}}
        {
        }

        const TShardId& Id() const override {
            return ShardId_;
        }

        TStringBuf ProjectId() const override {
            return TStringBuf("yasm");
        }

        TStringBuf ServiceName() const override {
            return TStringBuf("agent");
        }

        NDb::NModel::EPullProtocol Protocol() const override {
            return NDb::NModel::EPullProtocol::HTTP;
        }

        ui16 Port() const override {
            return YASM_PORT;
        }

        TMaybeTvmId TvmId() const override {
            return {};
        }

        TStringBuf UrlPath() const override {
            return TStringBuf("/json");
        }

        TDuration FetchInterval() const override {
            return NYasm::YASM_INTERVAL;
        }

        i32 GridSec() const override {
            return NDb::NModel::TServiceConfig::GRID_ABSENT;
        }

        bool IsAddTsArgs() const override {
            return false;
        }

        bool IsUseFqdn() const override {
            return true;
        }

        bool IsPull() const override {
            return true;
        }

        bool IsValid() const override {
            return true;
        }

        bool IsEqual(const IFetcherShard& other) const override {
            return Cluster_->Equals(*other.Cluster());
        }

        ui32 MaxResponseSizeBytes() const override {
            return RESPONSE_SIZE_LIMIT;
        }

        ui32 MaxMetricsPerUrl() const override {
            return METRIC_LIMIT;
        }

        const TClusterNode& Location() const override {
            return Location_;
        }

        TIntrusiveConstPtr<IFetcherCluster> Cluster() const override {
            return Cluster_;
        }

        EFetcherShardType Type() const override {
            return EFetcherShardType::YasmAgent;
        }

        TString ToString() const override {
            return {};
        }

        bool IsLocal() const override {
            return true;
        }

        void DumpConfig(IOutputStream* out) const override {
            *out << "Cluster: {" << Cluster_->Id().ToString() << "}\n";
            *out << "Location: {" << Location_ << "}\n";
        }

    private:
        TClusterNode Location_;
        TIntrusivePtr<IFetcherCluster> Cluster_;
        TShardId ShardId_{"yasm_agent_global", 0};
    };
} // namespace
    TFetcherShard CreateYasmAgentShard() {
        return TFetcherShard{new TYasmShard};
    }

    TFetcherShard CreateYasmAgentShard(TVector<TString> walleTags) {
        return TFetcherShard{new TYasmShard{std::move(walleTags)}};
    }

    TIntrusiveConstPtr<IFetcherCluster> CreateYasmAgentCluster() {
        return new TYasmCluster;
    }

} // namespace NSolomon::NFetcher
