#include <solomon/services/fetcher/lib/host_groups/host_resolver.h>

#include <solomon/libs/cpp/cloud/envoy/endpoint_v3_rpc.h>
#include <solomon/libs/cpp/cloud/instance_group/client.h>
#include <solomon/libs/cpp/dns/dns.h>
#include <solomon/libs/cpp/http/client/curl/client.h>
#include <solomon/protos/configs/rpc/rpc_config.pb.h>
#include <solomon/libs/cpp/cloud/instance_group/client.h>
#include <solomon/services/fetcher/lib/host_groups/host_resolver.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/folder/path.h>
#include <util/string/builder.h>
#include <util/generic/deque.h>
#include <util/generic/ptr.h>
#include <util/stream/file.h>
#include <util/string/builder.h>
#include <util/system/env.h>

using namespace NLastGetopt;
using namespace NSolomon;
using namespace NSolomon::NFetcher;
using namespace NMonitoring;

NMonitoring::TMetricRegistry Registry;
NSolomon::IHttpClientPtr HttpClient = NSolomon::CreateCurlClient({}, Registry);

TMetricRegistry DUMMY_REGISTRY;

enum EConductorEntityType {
    Tag = 0,
    Group = 1,
};

enum EYpEntityType {
    PodSet = 0,
    EndpointSet = 1,
    Label = 2,
};

class IHandler {
public:
    virtual ~IHandler() = default;

    void Usage(IOutputStream& out) const {
        out << Command();
        for (auto&& opt: Options()) {
            out << " " << opt;
        }
    }

    virtual const TString& Command() const = 0;
    virtual const TVector<TString>& Options() const = 0;
    virtual const TString& Description() const = 0;

    virtual void Handle(TDeque<TString> args, IOutputStream& out) = 0;

protected:
    void Print(const TUrls& urls, IOutputStream& out) {
        for (auto&& item: urls) {
            out << item.ToDebugString() << "\n----\n";
        }
    }
};

class THostListUrlHandler: public IHandler {
    const TString& Command() const override {
        return Command_;
    }

    const TVector<TString>& Options() const override {
        return Opts_;
    }

    const TString& Description() const override {
        return Descr_;
    }

    void Handle(TDeque<TString> args, IOutputStream& out) override {
        if (args.empty()) {
            ythrow TUsageException();
        }

        TString url {args.front()};
        args.pop_front();

        auto resolver = CreateHostUrlResolver(THostUrlResolverConfig{url, HttpClient.Get()}, DUMMY_REGISTRY);
        const auto urls = resolver->Resolve().ExtractValueSync().Extract();

        Print(urls, out);
    }

private:
    const TString Command_{"host-list-url"};
    const TVector<TString> Opts_{{"URL"}};
    const TString Descr_{"Get host list from the specified URL"};
};

class TNannyHandler: public IHandler {
public:
    TNannyHandler()
        : Command_{"nanny"}
        , Opts_{{"ENVIRONMENT", "SERVICE", "CFG_GROUPS", "PORT_SHIFT"}}
    {
    }

    const TString& Command() const override {
        return Command_;
    }

    const TVector<TString>& Options() const override {
        return Opts_;
    }

    const TString& Description() const override {
        return Descr_;
    }

    void Handle(TDeque<TString> args, IOutputStream& out) override {
        TNannyResolverConfig conf{HttpClient.Get()};

        if (args.size() > 4) {
            ythrow TUsageException();
        }

        if (args.size() >= 3) {
            conf.ClusterConfig.CfgGroups = {args[2]};
        }

        if (args.size() >= 4) {
            conf.ClusterConfig.PortShift = FromString<ui16>(args[3]);
        }

        conf.ClusterConfig.Env = FromString<ENannyEnv>(args[0]);
        conf.ClusterConfig.Service = args[1];

        auto resolver = CreateNannyResolver(conf, DUMMY_REGISTRY);
        const auto urls = resolver->Resolve().ExtractValueSync().Extract();

        Print(urls, out);
    }

private:
    const TString Command_{"host-list-url"};
    const TVector<TString> Opts_{{"URL"}};
    const TString Descr_{"Get host list from a Nanny service"};
};

class TQloudHandler: public IHandler {
public:
    TQloudHandler()
        : Command_{"qloud"}
        , Opts_{{"COMPONENT", "ENVIRONMENT", "APPLICATION", "PROJECT", "DEPLOYMENT"}}
    {
    }

    const TString& Command() const override {
        return Command_;
    }

    const TVector<TString>& Options() const override {
        return Opts_;
    }

    const TString& Description() const override {
        return Descr_;
    }

    void Handle(TDeque<TString> args, IOutputStream& out) override {
        if (args.size() != 5) {
            ythrow TUsageException();
        }

        TQloudResolverConfig conf;
        conf.ClusterConfig.Component = args[0];
        conf.ClusterConfig.Environment = args[1];
        conf.ClusterConfig.Application = args[2];
        conf.ClusterConfig.Project = args[3];
        conf.ClusterConfig.Deployment = args[4];
        conf.DnsClient = CreateDnsClient();

        auto resolver = CreateQloudResolver(conf);
        const auto urls = resolver->Resolve().ExtractValueSync().Extract();
        Print(urls, out);
    }


private:
    const TString Descr_{"Get host list from a Qloud application"};
    const TString Command_;
    const TVector<TString> Opts_;
};

class TConductorHandler: public IHandler {
public:
    TConductorHandler(EConductorEntityType type)
        : Type_{type}
        , Command_{type == EConductorEntityType::Tag ? "conductor-tag" : "conductor-group"}
        , Opts_{type == EConductorEntityType::Tag
            ? TVector<TString>{{"TAG"}}
            : TVector<TString>{{"GROUP"}}}
    {
    }

private:
    const TString& Command() const override {
        return Command_;
    }

    const TVector<TString>& Options() const override {
        return Opts_;
    }

    const TString& Description() const override {
        return Descr_;
    }

    void Handle(TDeque<TString> args, IOutputStream& out) override {
        if (args.empty()) {
            ythrow TUsageException();
        }

        auto group = args.front();
        args.pop_front();

        IHostGroupResolverPtr resolver;
        switch (Type_) {
            case EConductorEntityType::Tag:
                resolver = CreateConductorGroupResolver(TConductorResolverConfig{group, HttpClient.Get()}, DUMMY_REGISTRY);
                break;
            case EConductorEntityType::Group:
                resolver = CreateConductorTagResolver(TConductorResolverConfig{group, HttpClient.Get()}, DUMMY_REGISTRY);
                break;
        }

        const auto urls = resolver->Resolve().ExtractValueSync().Extract();
        Print(urls, out);
    }

private:
    const EConductorEntityType Type_;
    const TString Command_;
    const TVector<TString> Opts_;
    const TString Descr_{"Get host list from conductor"};
};

class TYpHandler: public IHandler {
public:
    TYpHandler(EYpEntityType et)
        : Command_{et == EYpEntityType::PodSet ? "yd-pod" : et == EYpEntityType::EndpointSet
            ? "yd-endpoint" : "yp-label"}
        , Type_{et}
        , Opts_{et == EYpEntityType::PodSet
            ? TVector<TString>{{"CLUSTER"}, {"POD_SET_ID"}}
            : et == EYpEntityType::EndpointSet
                ? TVector<TString>{{"CLUSTER"}, {"ENDPOINT_SET_ID"}}
                : TVector<TString>{{"CLUSTER"}, {"LABEL"}}
        }
    {
    }

    const TString& Command() const override {
        return Command_;
    }

    const TVector<TString>& Options() const override {
        return Opts_;
    }

    const TString& Description() const override {
        return Descr_;
    }

    void Handle(TDeque<TString> args, IOutputStream& out) override {
        Y_ENSURE_EX(args.size() == 2, TUsageException{});

        auto cluster = args[0];
        auto group = args[1];

        TYpResolverConfig config{HttpClient.Get()};
        config.ClusterConfig.Cluster = cluster;

        IHostGroupResolverPtr resolver;

        switch (Type_) {
            case EYpEntityType::PodSet:
                config.ClusterConfig.Type = TYpPodSet{.Id = group};
                resolver = CreateYdResolver(config, DUMMY_REGISTRY);
                break;
            case EYpEntityType::EndpointSet:
                config.ClusterConfig.Type = TYpEndpointSet{.Id = group};
                resolver = CreateYdResolver(config, DUMMY_REGISTRY);
                break;
            case EYpEntityType::Label:
                config.Token = GetYpToken();
                config.ClusterConfig.Type = TYpLabel{.Value = group};
                resolver = CreateYpResolver(config, DUMMY_REGISTRY);
                break;
            default:
                ythrow yexception() << "unknown YP entity type";
        }

        const auto urls = resolver->Resolve().ExtractValueSync().Extract();
        Print(urls, out);
    }

    TString GetYpToken() {
        TString token = GetEnv("YP_TOKEN");
        if (!token.empty()) {
            return token;
        }

        TFsPath tokenFile = JoinFsPaths(GetEnv("HOME"), ".yp", "token");
        if (tokenFile.Exists() && tokenFile.IsFile()) {
            try {
                token = TFileInput{tokenFile.GetPath()}.ReadLine();
            } catch (...) {
                Cerr << "Failed to read from " << tokenFile.GetPath() << ": " << CurrentExceptionMessage();
            }
        }

        Y_ENSURE(!token.empty(), "YP token must be specified either in ~/.yp/token or in the YP_TOKEN environment variable");
        return token;
    }

private:
    const TString Command_;
    const EYpEntityType Type_;
    const TVector<TString> Opts_;
    const TString Descr_{"Get YP group"};
};

class TInstanceGroupHandler: public IHandler {
public:
    const TString& Command() const override {
        return Command_;
    }

    const TVector<TString>& Options() const override {
        return Opts_;
    }

    const TString& Description() const override {
        return Descr_;
    }

    void Handle(TDeque<TString> args, IOutputStream &out) override {
        if (args.size() != 2) {
            ythrow TUsageException();
        }

        TString iamToken = GetEnv("IAM_TOKEN");
        if (iamToken.empty()) {
            Cerr << "environment variable IAM_TOKEN is empty" << Endl;
            ythrow TUsageException();
        }

        TString address = (args[0] == "prod")
                ? "instance-group.private-api.ycp.cloud.yandex.net"
                : "instance-group.private-api.ycp.cloud-preprod.yandex.net";
        TString groupId = args[1];

        yandex::solomon::config::rpc::TGrpcClientConfig grpcConfig;
        grpcConfig.add_addresses(address);

        TInstanceGroupResolverConfig config;
        config.ClusterConfig.GroupId = groupId;
        config.Client = NCloud::CreateInstanceGroupClient(
                grpcConfig,
                NCloud::CreateStaticTokenProvider(std::move(iamToken)),
                DUMMY_REGISTRY);

        auto resolver = CreateInstanceGroupResolver(config);
        auto urlsFuture = resolver->Resolve();
        const auto& urls = urlsFuture.GetValueSync();
        if (urls.Success()) {
            Print(urls.Value(), out);
        } else {
            Cerr << "cannot resolve instance group " << groupId << ": " << urls.Error().Message() << Endl;
        }
    }

private:
    const TString Command_{"instance-group"};
    const TString Descr_{"Get host list from instance group service"};
    const TVector<TString> Opts_{"{pre|prod}", "GROUP_ID"};
};

class TCloudDnsHandler: public IHandler {
public:
    const TString& Command() const override {
        return Command_;
    }

    const TVector<TString>& Options() const override {
        return Opts_;
    }

    const TString& Description() const override {
        return Descr_;
    }

    void Handle(TDeque<TString> args, IOutputStream &out) override {
        if (args.size() != 2) {
            ythrow TUsageException();
        }

        ECloudEnv env = FromString<ECloudEnv>(args[0]);
        TString name = args[1];

        std::map<ECloudEnv, TStringBuf> addressMap{
                {ECloudEnv::PROD, "xds.dns.cloud.yandex.net:18000"},
                {ECloudEnv::PREPROD, "xds.dns.cloud-preprod.yandex.net:18000"},
        };

        yandex::solomon::config::rpc::TGrpcClientConfig grpcConfig;
        for (auto& [_, address]: addressMap) {
            grpcConfig.add_addresses(TString{address});
        }

        TCloudDnsResolverConfig config;
        config.ProjectId = "yc.monitoring.cloud";
        config.ClusterConfig.Env = env;
        config.ClusterConfig.Name = name;
        config.Client = NCloud::NEnvoy::CreateEndpointClusterGrpc(grpcConfig, DUMMY_REGISTRY);

        auto resolver = CreateCloudDnsResolver(addressMap, config);
        auto urlsFuture = resolver->Resolve();
        const auto& urls = urlsFuture.GetValueSync();
        if (urls.Success()) {
            Print(urls.Value(), out);
        } else {
            Cerr << "cannot resolve name " << name << " in Cloud DNS: " << urls.Error().Message() << Endl;
        }
    }

private:
    const TString Command_{"cloud-dns"};
    const TString Descr_{"Get host list from Cloud DNS service"};
    const TVector<TString> Opts_{"{pre|prod}", "NAME"};
};

using IHandlerPtr = THolder<IHandler>;

void PrintUsageAndExit(const TMap<TString, IHandlerPtr>& handlers) {
    Cerr << "Usage: " << Endl;
    for (auto&& [_, handler]: handlers) {
        Cerr << "\t" << handler->Command();
        for (auto&& arg: handler->Options()) {
            Cerr << " " << arg;
        }

        Cerr << "\t--\t" << handler->Description();

        Cerr << Endl;
    }

    exit(1);
}

template <typename T, typename... Args>
auto Inserter(Args... args) {
    return [=] (TMap<TString, IHandlerPtr>& map) {
        IHandlerPtr p{new T{args...}};
        map.emplace(p->Command(), std::move(p));
    };
}

int main(int argc, const char** argv) {
    TOpts opts;
    TMap<TString, IHandlerPtr> handlers;

    Inserter<THostListUrlHandler>()(handlers);
    Inserter<TQloudHandler>()(handlers);
    Inserter<TConductorHandler>(EConductorEntityType::Group)(handlers);
    Inserter<TConductorHandler>(EConductorEntityType::Tag)(handlers);
    Inserter<TNannyHandler>()(handlers);
    Inserter<TYpHandler>(EYpEntityType::Label)(handlers);
    Inserter<TYpHandler>(EYpEntityType::PodSet)(handlers);
    Inserter<TYpHandler>(EYpEntityType::EndpointSet)(handlers);
    Inserter<TInstanceGroupHandler>()(handlers);
    Inserter<TCloudDnsHandler>()(handlers);

    TOptsParseResult res(&opts, argc, argv);
    auto&& vec = res.GetFreeArgs();
    TDeque<TString> args(vec.begin(), vec.end());
    if (!args.empty()) {
        const auto type = args.front();
        args.pop_front();

        auto handler = handlers.FindPtr(type);

        if (handler == nullptr) {
            ::PrintUsageAndExit(handlers);
        }

        try {
            (*handler)->Handle(args, Cout);
        } catch (TUsageException&) {
            Cerr << CurrentExceptionMessage() << Endl;
            ::PrintUsageAndExit(handlers);
        }

    } else {
        ::PrintUsageAndExit(handlers);
    }
}
