#include "host_resolver.h"
#include "host_resolver_iface.h"

#include <solomon/libs/cpp/cloud/envoy/cluster_load_assignment.h>

#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/svnversion/svnversion.h>

using namespace NThreading;
using namespace NMonitoring;

namespace NSolomon::NFetcher {
namespace {

TString MakeName(const TCloudDnsConfig& conf) {
    return TStringBuilder() << "cloud_dns:" << conf.Env << '/' << conf.Name;
}

class TCloudDnsResolver final: public IHostGroupResolver {
public:
    TCloudDnsResolver(TString projectId, TCloudDnsConfig config, NCloud::NEnvoy::IEndpointRpc* rpc) noexcept
        : ProjectId_{std::move(projectId)}
        , Config_{std::move(config)}
        , Name_{MakeName(Config_)}
        , Rpc_{rpc}
    {
    }

    TAsyncResolveResult Resolve() noexcept override {
        auto cb = [this] (NCloud::NEnvoy::TDescoveryAsyncResponse asyncResp) -> TResolveResult {
            try {
                auto val = asyncResp.ExtractValue();
                if (val.Success()) {
                    TVector<THostAndLabels> result;
                    std::set<TString> uniqueHosts;

                    envoy::service::discovery::v3::DiscoveryResponse& resp = val.Value();
                    for (const auto& resource: resp.resources()) {
                        auto endpoints = NCloud::NEnvoy::ParseClusterLoadAssignment(resource);
                        if (endpoints.Fail()) {
                            return TResolveError{endpoints.Error().MessageString()};
                        }

                        for (const auto& endpoint: endpoints.Value()) {
                            if (uniqueHosts.insert(endpoint.Hostname).second) {
                                result.emplace_back(endpoint.Hostname, Config_.Labels);
                            }
                        }
                    }

                    return result;
                }

                auto statusCode = static_cast<grpc::StatusCode>(val.Error().GRpcStatusCode);
                return TResolveError::FromGrpcStatus(statusCode, val.Error().Msg);
            } catch (...) {
                return TResolveError{CurrentExceptionMessage()};
            }
        };

        envoy::service::discovery::v3::DiscoveryRequest req;
        req.add_resource_names(Config_.Name);
        {
            auto* node = req.mutable_node();
            node->set_id(ProjectId_);
            node->set_user_agent_name("solomon-fetcher");
            node->set_user_agent_version(GetArcadiaLastChange());
        }

        return Rpc_->FetchEndpoints(req).Apply(std::move(cb));
    }

    const TString& Name() const override {
        return Name_;
    }

private:
    const TString ProjectId_;
    TCloudDnsConfig Config_;
    const TString Name_;
    NCloud::NEnvoy::IEndpointRpc* Rpc_;
};

} // namespace

IHostGroupResolverPtr CreateCloudDnsResolver(const std::map<ECloudEnv, TStringBuf>& addressMap, TCloudDnsResolverConfig config) {
    TString address;
    if (auto it = addressMap.find(config.ClusterConfig.Env); it != addressMap.end()) {
        address = it->second;
    } else {
        ythrow yexception() << "cannot create resolver, " << config.ProjectId << ' ' << MakeName(config.ClusterConfig);
    }

    NCloud::NEnvoy::IEndpointRpc* rpc = config.Client->Get(address);
    Y_ENSURE(rpc, "cannot create resolver, " << config.ProjectId << ' ' << MakeName(config.ClusterConfig));
    return new TCloudDnsResolver(std::move(config.ProjectId), std::move(config.ClusterConfig), rpc);
}

} // namespace NSolomon::NFetcher
