#include "host_resolver.h"

#include "http_base.h"

#include <util/string/builder.h>
#include <util/string/split.h>

namespace NSolomon::NFetcher {
    namespace {
        using namespace NThreading;
        using namespace NMonitoring;

        enum class EEntityType {
            Tag = 0,
            Group,
        };

        class TConductorResolver: public IHostGroupResolver, private THttpResolverBase {
        public:
            using TMethod = TAsyncResolveResult (TConductorResolver::*)();

            explicit TConductorResolver(TConductorResolverConfig config, TMetricRegistry& registry, EEntityType type)
                : THttpResolverBase{config}
                , Config_{std::move(config)}
                , Counters_{"conductor", registry}
            {
                switch (type) {
                    case EEntityType::Tag:
                        Method_ = &TConductorResolver::ResolveConductorTag;
                        Name_ = "conductor-tag:" + Config_.ClusterConfig.Name;
                        break;
                    case EEntityType::Group:
                        Method_ = &TConductorResolver::ResolveConductorGroup;
                        Name_ = "conductor-group:" + Config_.ClusterConfig.Name;
                        break;
                };

            }

            TAsyncResolveResult Resolve() noexcept override {
                return (this->*Method_)();
            }

            const TString& Name() const override {
                return Name_;
            }

            TAsyncResolveResult ResolveConductorGroup() noexcept {
                constexpr TStringBuf GROUP2HOSTS = "/api/groups2hosts/";
                TIntrusivePtr self{this};
                return DoRequest(BuildConductorRequest(GROUP2HOSTS, Config_.ClusterConfig.Name), [self] (const TString& result) {
                    return self->ParseGroupHosts(result);
                }, Counters_);
            }

            TAsyncResolveResult ResolveConductorTag() noexcept {
                constexpr TStringBuf TAG2HOSTS = "/api/tag2hosts/";
                TIntrusivePtr self{this};
                return DoRequest(BuildConductorRequest(TAG2HOSTS, Config_.ClusterConfig.Name), [self] (const TString& result) {
                    return self->ParseGroupHosts(result);
                }, Counters_);
            }

            TResolveResult ParseGroupHosts(const TString& raw) const noexcept try {
                TUrls result;
                for (TStringBuf line: StringSplitter(TStringBuf{raw}).Split('\n').SkipEmpty()) {
                    result.push_back(THostAndLabels::FromString(line, Config_.ClusterConfig.Labels));
                }

                return result;
            } catch (...) {
                return TResolveError{CurrentExceptionMessage()};
            }

        private:
            IRequestPtr BuildConductorRequest(TStringBuf path, const TString& group) {
                return Get(TStringBuilder() << Config_.ConductorUrl << path << group,
                    Headers({{"User-Agent", Config_.UserAgent}}));
            }

        private:
            TConductorResolverConfig Config_;
            TMethod Method_;
            TString Name_;
            TCounters Counters_;
        };
    }

    IHostGroupResolverPtr CreateConductorGroupResolver(const TConductorResolverConfig& config, TMetricRegistry& registry) {
        return ::MakeIntrusive<TConductorResolver>(config, registry, EEntityType::Group);
    }

    IHostGroupResolverPtr CreateConductorTagResolver(const TConductorResolverConfig& config, TMetricRegistry& registry) {
        return ::MakeIntrusive<TConductorResolver>(config, registry, EEntityType::Tag);
    }
}
