#include "config_parser.h"

#include <library/cpp/json/json_reader.h>
#include <library/cpp/monlib/metrics/labels.h>

#include <util/string/builder.h>

namespace NSolomon::NFetcher {
    using namespace NJson;
    using namespace NMonitoring;

    namespace {

    TString ParseStringOpional(NJson::TJsonValue::TMapType& map, TStringBuf key) {
        if (auto* raw = map.FindPtr(key); raw && !raw->IsNull()) {
            return raw->GetStringSafe();
        }

        return {};
    }

    TLabels ParseLabels(const TJsonValue& obj) {
        TLabels labels;
        if (obj["labels"].IsNull()) {
            return labels;
        }

        auto&& boxedLabels = obj["labels"].GetArraySafe();
        for (auto&& boxedLabel: boxedLabels) {
            auto&& label = boxedLabel.GetStringSafe();
            TStringBuf name, value;
            TStringBuf{label}.Split('=', name, value);
            Y_ENSURE(name && value, "Label has an incorrect format");
            labels.Add(name, value);
        }

        return labels;
    }

    TVector<TConductorConfig> ParseConductor(const TString& raw, TStringBuf fieldName) {
        TJsonValue json;
        Y_ENSURE(ReadJsonTree(raw, &json), "Invalid conductor group value");
        TVector<TConductorConfig> result;
        auto&& groups = json.GetArraySafe();
        for (auto&& boxedGroup: groups) {
            auto&& objMap = boxedGroup.GetMapSafe();
            auto&& group = objMap[fieldName].GetStringSafe();

            result.emplace_back(group, ParseLabels(boxedGroup));
        }

        return result;
    }
    } // namespace

    TVector<TConductorConfig> ParseConductorGroups(const TString& raw) {
        constexpr TStringBuf FIELD_NAME = "group";
        return ParseConductor(raw, FIELD_NAME);
    }

    TVector<TConductorConfig> ParseConductorTags(const TString& raw) {
        constexpr TStringBuf FIELD_NAME = "name";
        return ParseConductor(raw, FIELD_NAME);
    }

    TVector<THostPatternConfig> ParseHostPatternConfig(const TString& raw) {
        TJsonValue json;
        Y_ENSURE(ReadJsonTree(raw, &json), "not a valid JSON string: " << raw);

        auto&& groups = json.GetArraySafe();
        TVector<THostPatternConfig> result;
        for (auto&& boxedGroup: groups) {
            auto&& group = boxedGroup.GetMapSafe();
            const auto pattern = StripString(group["urlPattern"].GetStringSafe());
            const auto range = group["ranges"].GetStringSafe();
            const auto dc = group["dc"].GetStringSafe();
            const auto labels = ParseLabels(boxedGroup);

            result.emplace_back(pattern, range, labels, dc);
        }

        return result;
    }

    TVector<THostListUrlConfig> ParseHostUrlConfig(const TString& raw) {
        TJsonValue json;
        Y_ENSURE(ReadJsonTree(raw, &json), "not a valid JSON string: " << raw);

        auto&& groups = json.GetArraySafe();
        TVector<THostListUrlConfig> result;
        for (auto&& boxedGroup: groups) {
            auto&& group = boxedGroup.GetMapSafe();
            const auto url = StripString(group["url"].GetStringSafe());
            const auto ignorePorts = group["ignorePorts"].GetBooleanSafe();
            const auto labels = ParseLabels(boxedGroup);

            result.emplace_back(url, labels, ignorePorts);
        }

        return result;
    }

    TVector<TQloudConfig> ParseQloudConfig(const TString& raw) {
        TJsonValue json;
        ReadJsonTree(raw, &json, true);

        auto&& groups = json.GetArraySafe();
        TVector<TQloudConfig> result;
        for (auto&& boxedGroup: groups) {
            auto&& group = boxedGroup.GetMapSafe();

            TQloudConfig conf;
            conf.Component = group["component"].GetStringSafe();
            conf.Environment = group["environment"].GetStringSafe();
            conf.Application = group["application"].GetStringSafe();
            conf.Project = group["project"].GetStringSafe();
            conf.Deployment = group["deployment"].GetStringSafe();
            conf.Labels = ParseLabels(boxedGroup);
            result.push_back(std::move(conf));
        }

        return result;
    }

    TVector<TYpConfig> ParseYpConfig(const TString& raw) {
        TJsonValue json;
        ReadJsonTree(raw, &json, true);

        auto&& groups = json.GetArraySafe();
        TVector<TYpConfig> result;
        for (auto&& boxedGroup: groups) {
            auto&& group = boxedGroup.GetMapSafe();

            TYpConfig conf;
            if (auto* value = group.FindPtr("podSetId"); value && !value->GetString().Empty()) {
                conf.Type = TYpPodSet{.Id = value->GetString()};
            } else if (auto* value = group.FindPtr("ypLabel"); value && !value->GetString().Empty()) {
                conf.Type = TYpLabel{.Value = value->GetString(), .TvmLabel = ParseStringOpional(group, "tvmLabel")};
            } else if (auto* value = group.FindPtr("endpointSetId"); value && !value->GetString().Empty()) {
                conf.Type = TYpEndpointSet{.Id = value->GetString()};
            } else {
                ythrow yexception() << "Yp config is empty";
            }

            conf.Cluster = group["cluster"].GetStringSafe();

            conf.Labels = ParseLabels(boxedGroup);
            result.push_back(std::move(conf));
        }

        return result;
    }

    TVector<TNetworkConfig> ParseNetworkConfig(const TString& raw) {
        TJsonValue json;
        ReadJsonTree(raw, &json, true);

        TVector<TNetworkConfig> result;

        auto&& groups = json.GetArraySafe();
        for (auto&& boxedGroup: groups) {
            auto& conf = result.emplace_back();

            auto&& group = boxedGroup.GetMapSafe();
            conf.Network = group["network"].GetStringSafe();

            if (auto& portValue = group["port"]; !portValue.IsNull()) {
                conf.Port = portValue.GetUIntegerSafe();
            }

            conf.Labels = ParseLabels(boxedGroup);
        }

        return result;
    }

    TVector<TNannyConfig> ParseNannyConfig(const TString& raw) {
        TJsonValue json;
        ReadJsonTree(raw, &json, true);

        TVector<TNannyConfig> result;

        auto&& groups = json.GetArraySafe();
        for (auto&& boxedGroup: groups) {
            auto& conf = result.emplace_back();

            auto&& group = boxedGroup.GetMapSafe();
            conf.Service = group["service"].GetStringSafe();
            auto shift = group["portShift"].GetUIntegerSafe();

            Y_ENSURE(shift <= Max<ui16>(), "Port shift is out of range");
            conf.PortShift = shift;
            conf.Labels = ParseLabels(boxedGroup);
            conf.UseFetchedPort = group["useFetchedPort"].GetBooleanSafe();
            if (auto* value = group.FindPtr("env"); value && !value->GetString().Empty()) {
                conf.Env = FromString<ENannyEnv>(value->GetString());
            } else {
                conf.Env = ENannyEnv::PRODUCTION;
            }

            for (auto&& cfgGroup: group["cfgGroup"].GetArraySafe()) {
                conf.CfgGroups.push_back(cfgGroup.GetStringSafe());
            }
        }

        return result;
    }

    TVector<TInstanceGroupConfig> ParseInstanceGroupConfig(const TString& raw) {
        TJsonValue json;
        ReadJsonTree(raw, &json, true);

        TVector<TInstanceGroupConfig> result;

        auto&& groups = json.GetArraySafe();
        for (auto&& boxedGroup: groups) {
            auto& conf = result.emplace_back();

            auto&& group = boxedGroup.GetMapSafe();
            if (group.contains("instanceGroupId")) {
                conf.GroupId = group["instanceGroupId"].GetStringSafe();
            }

            if (group.contains("folderId")) {
                conf.FolderId = group["folderId"].GetStringSafe();
            }

            Y_ENSURE(conf.GroupId.empty() || conf.FolderId.empty(), "Only one of instanceGroupId and folderId may be present");

            conf.Labels = ParseLabels(boxedGroup);
        }

        return result;
    }

    TVector<TCloudDnsConfig> ParseCloudDnsConfig(const TString& raw) {
        TJsonValue json;
        ReadJsonTree(raw, &json, true);

        TVector<TCloudDnsConfig> result;
        for (auto&& boxedGroup: json.GetArraySafe()) {
            auto& conf = result.emplace_back();

            auto&& group = boxedGroup.GetMapSafe();
            (void) TryFromString(group["env"].GetString(), conf.Env);

            conf.Name = group["name"].GetStringSafe();
            Y_ENSURE(!conf.Name.empty(), "Name cannot be empty");

            conf.Labels = ParseLabels(boxedGroup);
        }

        return result;
    }

} // namespace NSolomon::NFetcher
