#include "factory.h"
#include "agents.h"
#include "host_resolver.h"

#include <solomon/libs/cpp/cloud/instance_group/client.h>

#include <library/cpp/monlib/metrics/metric_registry.h>

using namespace NMonitoring;
using namespace NThreading;


namespace NSolomon::NFetcher {
namespace {
    class TFakeUrlLocator: public IUrlLocator {
    public:
        bool IsLocal(const THostAndLabels& hl) const override {
            Y_UNUSED(hl);
            return false;
        }

        void SelectLocal(TVector<THostAndLabels>& hls) const override {
            hls = {};
        }
    };

    class TStubUrlLocator: public IUrlLocator {
    public:
        bool IsLocal(const THostAndLabels&) const override {
            return true;
        }

        void SelectLocal(TVector<THostAndLabels>&) const override {
        }
    };

    class THostResolverFactory: public IHostResolverFactory {
    public:
        explicit THostResolverFactory(THostResolverFactoryConfig conf) noexcept
            : Registry_{conf.Registry}
            , ClusterInfo_{conf.ClusterInfo}
            , DnsClient_{conf.DnsClient}
            , HttpClient_{conf.HttpClient}
            , ConductorConfig_{conf.ConductorConfig}
            , YpToken_{conf.YpToken}
            , InstanceGroupClient_{conf.InstanceGroupClient}
            , Locator_{std::move(conf.UrlLocator)}
            , EnvoyAddresses_{std::move(conf.EnvoyAddresses)}
            , EnvoyRpc_{std::move(conf.EnvoyRpc)}
        {
            Y_UNUSED(ClusterInfo_);
        }

        IHostGroupResolverPtr CreateConductorGroupResolver(TConductorConfig config) override {
            TConductorResolverConfig condConfig{std::move(config), HttpClient_.Get()};
            condConfig.SetClientConfig(ConductorConfig_);
            return NSolomon::NFetcher::CreateConductorGroupResolver(condConfig, Registry_);
        }

        IHostGroupResolverPtr CreateConductorTagResolver(TConductorConfig config) override {
            return NSolomon::NFetcher::CreateConductorTagResolver({std::move(config), HttpClient_.Get()}, Registry_);
        }

        IHostGroupResolverPtr CreateHostPatternResolver(THostPatternConfig config) override {
            return NSolomon::NFetcher::CreateHostPatternResolver({std::move(config)});
        }

        IHostGroupResolverPtr CreateHostUrlResolver(THostListUrlConfig config) override {
            return NSolomon::NFetcher::CreateHostUrlResolver({std::move(config), HttpClient_.Get()}, Registry_);
        }

        IHostGroupResolverPtr CreateQloudResolver(TQloudConfig config) override {
            TQloudResolverConfig c;
            c.SetClusterConfig(std::move(config))
                .SetDnsClient(DnsClient_);

            return NSolomon::NFetcher::CreateQloudResolver(std::move(c));
        }

        IHostGroupResolverPtr CreateYpResolver(TYpConfig config) override {
            TYpResolverConfig c{HttpClient_.Get()};
            c.ClusterConfig = std::move(config);
            c.Token = YpToken_;
            if (std::holds_alternative<TYpPodSet>(c.ClusterConfig.Type)) {
                return CreateYdResolver(std::move(c), Registry_);
            } else if (std::holds_alternative<TYpEndpointSet>(c.ClusterConfig.Type)) {
                return CreateYdResolver(std::move(c), Registry_);
            } else if (std::holds_alternative<TYpLabel>(c.ClusterConfig.Type)) {
                return NSolomon::NFetcher::CreateYpResolver(std::move(c), Registry_);
            } else {
                ythrow yexception() << "Yp config is empty";
            }
        }

        IHostGroupResolverPtr CreateNetworkResolver(TNetworkConfig config) override {
            return NSolomon::NFetcher::CreateNetworkResolver({std::move(config)});
        }

        IHostGroupResolverPtr CreateNannyResolver(TNannyConfig config) override {
            TNannyResolverConfig c{HttpClient_.Get()};
            c.ClusterConfig = std::move(config);
            return NSolomon::NFetcher::CreateNannyResolver(std::move(c), Registry_);
        }

        IHostGroupResolverPtr CreateInstanceGroupResolver(TInstanceGroupConfig config) override {
            TInstanceGroupResolverConfig c;
            c.ClusterConfig = std::move(config);
            c.Client = InstanceGroupClient_;

            return NSolomon::NFetcher::CreateInstanceGroupResolver(c);
        }

        IHostGroupResolverPtr CreateAgentGroupResolver(TString name, TVector<THostAndLabels> hosts) override {
            return NSolomon::NFetcher::CreateAgentGroupResolver(std::move(name), std::move(hosts), Locator_);
        }

        IHostGroupResolverPtr CreateYasmAgentGroupResolver(TVector<TString> tags, TVector<TString> conductorTags) override {
            Y_UNUSED(tags);
            Y_UNUSED(conductorTags);
            /*
             May be useful later

            EDc dc{EDc::UNKNOWN};
            if (ClusterInfo_.OperationMode() == EOperationMode::Local) {
                dc = ClusterInfo_.Dc();
            }

            TWalleResolverConfig c{HttpClient_, dc};
            c.Tags = std::move(tags);

            if (conductorTags.empty()) {
                return NSolomon::NFetcher::CreateYasmAgentGroupResolver(std::move(c), Registry_, Locator_);
            } else {
                // XXX only one tag at a time is supported at the moment
                Y_VERIFY_DEBUG(conductorTags.size() == 1);
                TConductorResolverConfig conductor{conductorTags[0], HttpClient_};
                return NSolomon::NFetcher::CreateYasmAgentGroupResolver(std::move(c), std::move(conductor), Registry_, Locator_);
            }
            */

            return CreateYasmAgentHostsFileResolver(Locator_, &Registry_);
        }

        IHostGroupResolverPtr CreateCloudDnsResolver(TString projectId, struct TCloudDnsConfig config) override {
            TCloudDnsResolverConfig resolverConfig{
                    .ProjectId = projectId,
                    .ClusterConfig = std::move(config),
                    .Client = EnvoyRpc_,
            };
            return NSolomon::NFetcher::CreateCloudDnsResolver(EnvoyAddresses_, std::move(resolverConfig));
        }

    private:
        TMetricRegistry& Registry_;
        const TClusterInfo& ClusterInfo_;
        IDnsClientPtr DnsClient_;
        IHttpClientPtr HttpClient_;

        TConductorClientConfig ConductorConfig_;
        TString YpToken_;
        TIntrusivePtr<NCloud::IInstanceGroupClient> InstanceGroupClient_;
        IUrlLocatorPtr Locator_;

        std::map<ECloudEnv, TStringBuf> EnvoyAddresses_;
        NCloud::NEnvoy::IEndpointClusterRpcPtr EnvoyRpc_;
    };
} // namespace
    IUrlLocatorPtr FakeUrlLocator() {
        return new TFakeUrlLocator;
    }

    IUrlLocatorPtr StubUrlLocator() {
        return new TStubUrlLocator;
    }

    IHostResolverFactoryPtr CreateHostResolverFactory(THostResolverFactoryConfig conf) {
        return new THostResolverFactory{std::move(conf)};
    }
} // namespace NSolomon::NFetcher
