#pragma once

#include <library/cpp/monlib/metrics/labels.h>
#include <library/cpp/ipv6_address/ipv6_address.h>

#include <util/generic/string.h>
#include <util/generic/maybe.h>
#include <util/digest/multi.h>

#include <util/string/builder.h>
#include <util/string/split.h>

#include <optional>
#include <utility>

class IOutputStream;

namespace NSolomon::NFetcher {
    // NB: naming is confusing: TIpv6Address in fact can hold both IPv4 and IPv6 addresses
    using TMaybeIp = TMaybe<TIpv6Address>;
    using TMaybeTvmId = std::optional<ui32>;
    using TMaybeDuration = std::optional<TDuration>;

    struct THostAndLabels {
        static constexpr ui16 DEFAULT_PORT = 0;

        THostAndLabels(TString host, NMonitoring::TLabels labels = {})
            : Host{std::move(host)}
            , Labels{std::move(labels)}
        {
        }

        THostAndLabels(TIpv6Address address, NMonitoring::TLabels labels = {})
            : IpAddress{address}
            , Labels{std::move(labels)}
        {
        }

        THostAndLabels(const TString& host, ui16 port, TMaybeIp ipAddress, NMonitoring::TLabels labels = {}, i16 portShift = 0)
            : Host{host}
            , Port{port}
            , PortShift{portShift}
            , IpAddress{ipAddress}
            , Labels{std::move(labels)}
        {
            Y_ENSURE(!host.empty() || ipAddress.Defined(), "Either host or IP address must be present and not empty");
        }

        THostAndLabels(const THostAndLabels&) = default;
        THostAndLabels& operator=(const THostAndLabels&) = default;

        THostAndLabels(THostAndLabels&&) = default;
        THostAndLabels& operator=(THostAndLabels&&) = default;

        static THostAndLabels FromString(TStringBuf str, const NMonitoring::TLabels& additionalLabels = {});

        TString ToString() const;
        TString ToDebugString() const;

        ui64 UrlHash() const;

        TString Host;
        ui16 Port {0};
        i16 PortShift {0};

        TMaybeIp IpAddress;
        TMaybeTvmId TvmDestId;

        NMonitoring::TLabels Labels;
    };

    inline TVector<THostAndLabels> HostAndLabelsListFromString(TStringBuf raw) {
        TVector<THostAndLabels> result;
        StringSplitter(raw).Split('\n').SkipEmpty().Consume([&result] (TStringBuf line) {
            result.push_back(THostAndLabels::FromString(line));
        });

        return result;
    }

    template <typename TContainer>
    TString HostAndLabelsListToString(TContainer&& hostAndLabelsList) {
        static_assert(std::is_same<typename std::decay<TContainer>::type::value_type, THostAndLabels>::value,
            "Expected a container of THostAndLabels items here");

        TStringBuilder result;

        for (auto&& hostAndLabels: hostAndLabelsList) {
            result << hostAndLabels.ToString() << '\n';
        }

        return result;
    }


    IOutputStream& operator<<(IOutputStream& os, const THostAndLabels& hostAndLabels);

    inline bool operator==(const THostAndLabels& lhs, const THostAndLabels& rhs) {
        return lhs.Host == rhs.Host
            && lhs.Port == rhs.Port
            && lhs.PortShift == rhs.PortShift
            && lhs.IpAddress == rhs.IpAddress
            && lhs.TvmDestId == rhs.TvmDestId
            && lhs.Labels == rhs.Labels;
    }

} // namespace NSolomon::NFetcher

template <>
struct THash<NSolomon::NFetcher::THostAndLabels> {
    inline size_t operator()(const NSolomon::NFetcher::THostAndLabels& h) const noexcept {
        auto hash = MultiHash(
                h.Host,
                h.Port,
                h.PortShift,
                h.Labels
            );

        if (h.IpAddress) {
            hash = CombineHashes(hash, THash<TIpv6Address>{}(*h.IpAddress));
        }

        return hash;
    }
};
