#include "host_resolver.h"
#include "host_resolver_iface.h"
#include "http_base.h"

#include <library/cpp/string_utils/url/url.h>

#include <util/string/split.h>
#include <util/string/strip.h>

namespace NSolomon::NFetcher {
    namespace {
        using namespace NThreading;
        using namespace NMonitoring;

        TString MakeMetricName(TStringBuf url) {
            TStringBuf scheme, host;
            ui16 port{0};
            if (!TryGetSchemeHostAndPort(url, scheme, host, port)) {
                return "UNKNOWN";
            }

            TStringBuilder sb;
            sb << host;

            if (port != 0) {
                sb << '_' << port;
            }

            return sb;
        }

        class THostUrlResolver: THttpResolverBase, public IHostGroupResolver {
        public:
            explicit THostUrlResolver(const THostUrlResolverConfig& config, TMetricRegistry& registry)
                : THttpResolverBase{config}
                , Config_{config}
                , Url_{Config_.ClusterConfig.Url}
                , Counters_{MakeMetricName(config.ClusterConfig.Url), registry}
            {
            }

            TAsyncResolveResult Resolve() noexcept override {
                TIntrusivePtr self{this};
                return DoRequest(Get(Url_), [self] (const TString& result) {
                    return self->ParseUrls(result);
                }, Counters_);
            }

            const TString& Name() const override {
                return Url_;
            }

            TResolveResult ParseUrls(const TString& raw) const noexcept try {
                TUrls result;
                StringSplitter(TStringBuf{raw})
                    .Split('\n')
                    .SkipEmpty()
                    .Consume([&] (TStringBuf l) {
                        auto line = StripString(l);
                        result.push_back(THostAndLabels::FromString(line, Config_.ClusterConfig.Labels));

                        if (Config_.ClusterConfig.IgnorePorts) {
                            result.back().Port = THostAndLabels::DEFAULT_PORT;
                        }
                    });

                return TResolveResult::FromValue(result);
            } catch (...) {
                return TResolveResult::FromError(CurrentExceptionMessage());
            }

        private:
            THostUrlResolverConfig Config_;
            const TString& Url_;
            TCounters Counters_;
        };
    }

    IHostGroupResolverPtr CreateHostUrlResolver(const THostUrlResolverConfig& config, TMetricRegistry& registry) {
        return ::MakeIntrusive<THostUrlResolver>(config, registry);
    }
} // namespace NSolomon::NFetcher
