#include "host_resolver.h"
#include "host_resolver_iface.h"
#include "http_base.h"

#include <util/string/builder.h>
#include <util/string/split.h>
#include <util/string/printf.h>

#include <contrib/libs/re2/re2/re2.h>

namespace NSolomon::NFetcher {
    namespace {
        using namespace NThreading;

        class THostPatternResolver: public IHostGroupResolver {
            static const re2::RE2 RANGE_PATTERN;

        public:
            THostPatternResolver(THostPatternResolverConfig config)
                : Config_{std::move(config)}
                , Pattern_{Config_.ClusterConfig.Pattern}
                , Name_{Config_.ClusterConfig.Pattern}
            {
                // validate pattern
                if (!IsValidPattern()) {
                    Error_ = TResolveError(TString{"Pattern is invalid: "} + Pattern_);
                }

                // parse ranges
                for (auto range: StringSplitter(Config_.ClusterConfig.Ranges).Split(' ').SkipEmpty()) {
                    const auto tok = range.Token();

                    int from, to = -1;
                    TString toStr;

                    if (!RE2::FullMatch(re2::StringPiece{static_cast<std::string>(tok)}, RANGE_PATTERN, &from, &toStr)) {
                        Error_ = TResolveError(TString{"Range is invalid: "} + tok);
                    } else if (from < 0) {
                        Error_ = TResolveError(TString{"Range is invalid: "} + tok);
                    }

                    if (Error_) {
                        continue;
                    }

                    if (!toStr.empty()) {
                        TStringBuf sb{toStr.data(), toStr.size()};
                        // skip the dash
                        sb.Skip(1);
                        to = FromString<int>(sb);
                    }

                    if (to == -1) {
                        Ranges_.push_back(std::make_pair(from, from));
                    }

                    Ranges_.push_back(std::make_pair(from, to));
                }
            }

            bool IsValidPattern() const noexcept {
                const auto formatStartPosition = Pattern_.find('%');
                if (formatStartPosition == Pattern_.npos) {
                    return true;
                }

                // skip padding, find formatting
                const auto substTypePosition = Pattern_.find_first_not_of("0123456789", formatStartPosition + 1);

                static const TString ALLOWED_TYPES = "dui";

                return substTypePosition != TString::npos
                       // only type specifiers for integer types
                       && ALLOWED_TYPES.find(Pattern_[substTypePosition]) != TString::npos
                       // we allow only one subtitution
                       && Pattern_.find('%', substTypePosition) == TString::npos;
            }

            TAsyncResolveResult Resolve() noexcept override try {
                if (Error_) {
                    return MakeFuture<TResolveResult>(TResolveResult{std::move(*Error_)});
                }

                if (Pattern_.find('%') == Pattern_.npos) {
                    return MakeFuture<TResolveResult>(TResolveResult{{THostAndLabels::FromString(Pattern_, Config_.ClusterConfig.Labels)}});
                }

                TUrls result;
                for (auto&& [left, right]: Ranges_) {
                    for (auto i = left; i <= right; ++i) {
                        const auto& str = Sprintf(Pattern_.c_str(), i);
                        result.push_back(THostAndLabels::FromString(str, Config_.ClusterConfig.Labels));
                    }
                }

                return MakeFuture<TResolveResult>(result);
            } catch (...) {
                return MakeFuture<TResolveResult>(TResolveError(CurrentExceptionMessage()));
            }


            const TString& Name() const override {
                return Name_;
            }

            std::optional<TString> CacheKey() const override {
                return {};
            }

        private:
            THostPatternResolverConfig Config_;
            const TString& Pattern_;
            TString Name_;
            std::optional<TResolveError> Error_;
            TVector<std::pair<int, int>> Ranges_;
        };

        const re2::RE2 THostPatternResolver::RANGE_PATTERN = re2::RE2("(\\d+)(-\\d+)?");

    }

    IHostGroupResolverPtr CreateHostPatternResolver(THostPatternResolverConfig config) {
        return ::MakeIntrusive<THostPatternResolver>(std::move(config));
    }
} // namespace NSolomon::NFetcher
