#pragma once

#include "host_and_labels.h"
#include "resolver_config.h"

#include <solomon/services/fetcher/config/resolver_config.pb.h>
#include <solomon/services/fetcher/lib/fetcher_shard.h>

#include <solomon/libs/cpp/cloud/envoy/endpoint_v3_rpc.h>
#include <solomon/libs/cpp/config/units.h>
#include <solomon/libs/cpp/dns/dns.h>
#include <solomon/libs/cpp/http/client/http.h>

#include <library/cpp/monlib/metrics/fwd.h>
#include <library/cpp/threading/future/future.h>

#include <utility>

namespace NSolomon::NCloud {
    struct IInstanceGroupClient;

} // namespace NSolomon::NCloud

namespace NSolomon::NFetcher {
    using TUrls = TVector<THostAndLabels>;

    struct THttpResolverConfig {
        static constexpr auto READ_TIMEOUT = TDuration::Minutes(1);

        explicit THttpResolverConfig(IHttpClient* client)
            : Client{client}
        {
        }

        void FromProto(const THttpClientConfig& proto) {
            Retries = proto.GetRetries();
            if (proto.HasTimeout()) {
                Timeout = FromProtoTime(proto.GetTimeout());
            }
        }

        ui8 Retries{1};
        TDuration Timeout{READ_TIMEOUT};
        TDuration ConnectTimeout{TDuration::Seconds(20)};
        TString UserAgent{"SolomonFetcher"};
        IHttpClient* Client{nullptr};
    };

    struct TConductorResolverConfig: THttpResolverConfig {
        TConductorResolverConfig() = default;

        explicit TConductorResolverConfig(TString name, IHttpClient* client)
            : THttpResolverConfig{client}
            , ClusterConfig{std::move(name)}
        {
        }

        TConductorResolverConfig(TConductorConfig&& cluster, IHttpClient* client)
            : THttpResolverConfig{client}
            , ClusterConfig{std::move(cluster)}
        {
        }

        TConductorResolverConfig& SetClientConfig(const TConductorClientConfig& clientConf) {
            if (clientConf.HasHttpClient()) {
                FromProto(clientConf.GetHttpClient());
            }

            if (auto&& url = clientConf.GetConductorUrl(); !url.empty()) {
                ConductorUrl = url;
            }

            return *this;
        }

        TConductorResolverConfig& SetClusterConfig(TConductorConfig clusterConf) {
            ClusterConfig = std::move(clusterConf);
            return *this;
        }

        TConductorResolverConfig& SetUrl(const TString& url) {
            if (!url.empty()) {
                ConductorUrl = url;
            }

            return *this;
        }

        TString ConductorUrl{"http://c.yandex-team.ru"};
        TConductorConfig ClusterConfig;
    };

    struct THostPatternResolverConfig {
        THostPatternResolverConfig() = default;

        explicit THostPatternResolverConfig(TString pattern, TString range)
            : ClusterConfig{std::move(pattern), std::move(range)}
        {
        }

        THostPatternResolverConfig(THostPatternConfig&& clusterConfig)
            : ClusterConfig{std::move(clusterConfig)}
        {
        }

        THostPatternConfig ClusterConfig;
    };

    struct THostUrlResolverConfig: THttpResolverConfig {
        using THttpResolverConfig::THttpResolverConfig;

        THostUrlResolverConfig& SetHttpConfig(const THttpClientConfig& httpConf) {
            FromProto(httpConf);
            return *this;
        }

        THostUrlResolverConfig& SetClusterConfig(THostListUrlConfig clusterConf) {
            ClusterConfig = std::move(clusterConf);
            return *this;
        }

        THostUrlResolverConfig(THostListUrlConfig&& clusterConfig, IHttpClient* client)
            : THttpResolverConfig{client}
            , ClusterConfig{std::move(clusterConfig)}
        {
        }

        THostListUrlConfig ClusterConfig;
    };

    struct TQloudResolverConfig {
        TQloudResolverConfig() = default;
        TQloudResolverConfig& SetClusterConfig(TQloudConfig clusterConf) {
            ClusterConfig = std::move(clusterConf);
            return *this;
        }

        TQloudResolverConfig& SetDnsClient(IDnsClientPtr dnsClient) {
            DnsClient = std::move(dnsClient);
            return *this;
        }

        TQloudConfig ClusterConfig;
        IDnsClientPtr DnsClient;
    };

    struct TYpResolverConfig: THttpResolverConfig {
        using THttpResolverConfig::THttpResolverConfig;

        TString Token;
        TYpConfig ClusterConfig;
    };

    struct TNetworkResolverConfig {
        TNetworkConfig ClusterConfig;
    };

    struct TNannyResolverConfig: THttpResolverConfig {
        using THttpResolverConfig::THttpResolverConfig;
        TNannyConfig ClusterConfig;
    };

    struct TInstanceGroupResolverConfig {
        TInstanceGroupResolverConfig();
        ~TInstanceGroupResolverConfig();

        TInstanceGroupConfig ClusterConfig;
        TIntrusivePtr<NCloud::IInstanceGroupClient> Client;
    };

    struct TCloudDnsResolverConfig {
        TString ProjectId;
        TCloudDnsConfig ClusterConfig;
        NCloud::NEnvoy::IEndpointClusterRpcPtr Client;
    };

    // For backward compatibility
    IHostGroupResolverPtr CreateConductorGroupResolver(const TConductorResolverConfig& config, NMonitoring::TMetricRegistry& registry);
    IHostGroupResolverPtr CreateConductorTagResolver(const TConductorResolverConfig& config, NMonitoring::TMetricRegistry& registry);
    IHostGroupResolverPtr CreateHostPatternResolver(THostPatternResolverConfig config);
    IHostGroupResolverPtr CreateHostUrlResolver(const THostUrlResolverConfig& config, NMonitoring::TMetricRegistry& registry);
    IHostGroupResolverPtr CreateQloudResolver(TQloudResolverConfig config);
    IHostGroupResolverPtr CreateYdResolver(TYpResolverConfig config, NMonitoring::TMetricRegistry& registry);
    IHostGroupResolverPtr CreateYpResolver(const TYpResolverConfig& config, NMonitoring::TMetricRegistry& registry);
    IHostGroupResolverPtr CreateNetworkResolver(TNetworkResolverConfig config);
    IHostGroupResolverPtr CreateNannyResolver(TNannyResolverConfig config, NMonitoring::TMetricRegistry& registry);
    IHostGroupResolverPtr CreateInstanceGroupResolver(TInstanceGroupResolverConfig config);
    IHostGroupResolverPtr CreateCloudDnsResolver(const std::map<ECloudEnv, TStringBuf>& addressMap, TCloudDnsResolverConfig config);

} // namespace NSolomon::NFetcher
