#pragma once

#include <solomon/libs/cpp/error_or/error_or.h>

#include <library/cpp/threading/future/future.h>
#include <library/cpp/http/misc/httpcodes.h>

#include <contrib/libs/grpc/include/grpcpp/impl/codegen/status_code_enum.h>

namespace NSolomon::NFetcher {
    struct THostAndLabels;

    struct TResolveError: public TGenericError {
        enum EType {
            Permanent = 0,
            Transient,
        };

        explicit TResolveError(TString message, EType type = Permanent) noexcept
            : TGenericError{std::move(message)}
            , Type_{type}
        {
        }

        explicit TResolveError(TGenericError err, EType type = Permanent) noexcept
            : TGenericError{std::move(err)}
            , Type_{type}
        {
        }

        static TResolveError FromHttpStatus(HttpCodes code, TString message = {}) {
            if (IsServerError(code) || code == HTTP_TOO_MANY_REQUESTS) {
                return TResolveError{std::move(message), Transient};
            }
            return TResolveError{std::move(message), Permanent};
        }

        static TResolveError FromGrpcStatus(grpc::StatusCode code, TString message = {}) {
            switch (code) {
                case grpc::DEADLINE_EXCEEDED:
                case grpc::RESOURCE_EXHAUSTED:
                case grpc::INTERNAL:
                case grpc::UNAVAILABLE:
                    return TResolveError{std::move(message), Transient};

                default:
                    return TResolveError{std::move(message), Permanent};
            }
        }

        bool IsTransient() const {
            return Type_ == Transient;
        };

        EType Type() const {
            return Type_;
        }

    private:
        EType Type_;
    };

    using TResolveResult = TErrorOr<TVector<THostAndLabels>, TResolveError>;
    using TAsyncResolveResult = NThreading::TFuture<TResolveResult>;

    class IHostGroupResolver: public TThrRefBase {
    public:
        virtual TAsyncResolveResult Resolve() noexcept = 0;
        virtual const TString& Name() const = 0;

        virtual std::optional<TString> CacheKey() const {
            return Name();
        }
    };

    using IHostGroupResolverPtr = TIntrusivePtr<IHostGroupResolver>;
} // namespace NSolomon::NFetcher
