#pragma once

#include "host_resolver_iface.h"
#include "host_resolver.h"
#include "http_counters.h"

#include <solomon/libs/cpp/http/client/http.h>

#include <library/cpp/http/misc/httpcodes.h>
#include <library/cpp/threading/future/future.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/generic/string.h>
#include <util/string/builder.h>

#include <type_traits>

namespace NSolomon::NFetcher {
    class THttpResolverBase {
    protected:
        using TCounters = THttpCounters;
        THttpResolverBase(const THttpResolverConfig& conf)
            : Client_{conf.Client}
            , RequestOpts_{
                .ConnectTimeout = conf.ConnectTimeout,
                .ReadTimeout = conf.Timeout,
                .Retries = conf.Retries,
            }
        {
            Y_VERIFY_DEBUG(Client_);
        }

        template <typename TParser>
        auto DoRequest(IRequestPtr req, TParser p, TCounters counters)
            -> NThreading::TFuture<std::invoke_result_t<TParser, TString>>
        {
            static_assert(std::is_invocable_v<TParser, TString>);
            using T = std::invoke_result_t<TParser, TString>;
            counters.StartRequest();

            auto promise = NThreading::NewPromise<T>();
            auto cb = [=, parser = std::move(p), url = TString{req->Url()}] (IHttpClient::TResult result) mutable {
                if (!result.Success()) {
                    TString msg = TStringBuilder() << "Request to " << url << " failed: " << result.Error().Message();
                    promise.SetValue(TResolveError{std::move(msg), TResolveError::Transient});
                } else if (auto code = result.Value()->Code(); code != HTTP_OK) {
                    counters.CompleteRequest(code);
                    TString msg = TStringBuilder() << "Request to " << url << " failed with code " << code;
                    promise.SetValue(TResolveError::FromHttpStatus(code, std::move(msg)));
                } else {
                    auto r = parser(result.Value()->ExtractData());
                    promise.SetValue(std::move(r));
                }
            };

            Client_->Request(std::move(req), std::move(cb), RequestOpts_);

            return promise.GetFuture();
        }

    private:
        IHttpClient* Client_{nullptr};
        TRequestOpts RequestOpts_;
    };
}
