#include "host_resolver.h"
#include "host_resolver_iface.h"

#include <solomon/libs/cpp/cloud/instance_group/client.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

namespace NSolomon::NFetcher {
    TInstanceGroupResolverConfig::TInstanceGroupResolverConfig() = default;
    TInstanceGroupResolverConfig::~TInstanceGroupResolverConfig() = default;

namespace {
    using namespace NThreading;
    using namespace NMonitoring;

    TString MakeName(const TInstanceGroupConfig& conf) {
        if (!conf.GroupId.empty()) {
            return TStringBuilder() << "instance_group:" << conf.GroupId;
        } else if (!conf.FolderId.empty()) {
            return TStringBuilder() << "instance_group:" << conf.FolderId;
        }

        ythrow yexception() << "Invalid config: has neither instanceGroupId nor folderId";
    }

    class TInstanceGroupResolver final: public IHostGroupResolver {
    public:
        TInstanceGroupResolver(const TInstanceGroupResolverConfig& config, NCloud::IInstanceGroupClientPtr client)
            : Config_{std::move(config)}
            , Name_{MakeName(Config_.ClusterConfig)}
            , Client_{std::move(client)}
        {
        }

        TAsyncResolveResult Resolve() noexcept override {
            if (!Client_) {
                return MakeFuture(TResolveResult::FromValue());
            }

            auto cb = [this] (auto result) {
                auto val = result.ExtractValue();

                if (val.Success()) {
                    TUrls result;
                    for (auto&& instance: val.Value().Instances) {
                        result.emplace_back(instance.Fqdn, Config_.ClusterConfig.Labels);
                    }

                    return TResolveResult::FromValue(result);
                }

                // TODO(msherbakov):Get more info from the instance groups client so that
                // we can distinguish between transient and permanent errors
                return TResolveResult::FromError(val.Error().MessageString());
            };

            if (Config_.ClusterConfig.GroupId) {
                return Client_->ListInstances(Config_.ClusterConfig.GroupId).Apply(std::move(cb));
            } else {
                return Client_->ListInstancesForFolder(Config_.ClusterConfig.FolderId).Apply(std::move(cb));
            }
        }

        const TString& Name() const override {
            return Name_;
        }

    private:
        TInstanceGroupResolverConfig Config_;
        const TString Name_;
        NCloud::IInstanceGroupClientPtr Client_;
    };
} // namespace

IHostGroupResolverPtr CreateInstanceGroupResolver(TInstanceGroupResolverConfig config) {
    return new TInstanceGroupResolver(std::move(config), std::move(config.Client));
}

} // namespace NSolomon::NFetcher
