#include "host_resolver.h"
#include "http_base.h"

#include <solomon/libs/cpp/error_or/error_or.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/string/builder.h>
#include <util/system/spinlock.h>

namespace NSolomon::NFetcher {
namespace {
    using namespace NThreading;
    using namespace NMonitoring;

    struct TWaitContext: public TAtomicRefCount<TWaitContext> {
        TWaitContext() { }
        ~TWaitContext() {
            if (!(Promise.HasValue() || Promise.HasException())) {
                Promise.SetValue(std::move(Result));
            }
        }

        void Fail(TResolveError&& e) {
            Promise.SetValue(std::move(e));
        }

        void Complete(TAsyncResolveResult f) {
            auto r = f.ExtractValue();
            if (r.Fail()) {
                Promise.SetValue(std::move(r));
                return;
            }

            auto val = r.Extract();

            auto g = Guard(Lock);
            Copy(
                std::make_move_iterator(val.begin()),
                std::make_move_iterator(val.end()),
                std::back_inserter(Result)
            );
        }

        TUrls Result;
        TAdaptiveLock Lock;

        TPromise<TResolveResult> Promise{NewPromise<TResolveResult>()};
    };

    constexpr auto API_URL = "http://nanny.yandex-team.ru/v2/services/";
    constexpr auto ADMIN_API_URL = "http://adm-nanny.yandex-team.ru/v2/services/";
    constexpr auto GENCFG_API_URL = "http://api.gencfg.yandex-team.ru/";

    const TVector<std::pair<TString, TString>> HEADERS {
        {"Accept", "application/json"},
    };

    enum class ENetworkSettings {
        NoIsolation = 0,
        MtnEnabled = 1,
    };

    struct TNannyHost {
        TString Hostname;
        TString ContainerHostname;
        ENetworkSettings NetworkSettings{ENetworkSettings::NoIsolation};
        ui16 Port{0};

        static TNannyHost FromJson(const NJson::TJsonValue& obj) {
            TNannyHost host;
            auto&& network = obj["network_settings"].GetStringSafe();
            if (network == TStringBuf("MTN_ENABLED")) {
                host.NetworkSettings = ENetworkSettings::MtnEnabled;
            }

            host.ContainerHostname = obj["container_hostname"].GetStringSafe();
            host.Hostname = obj["hostname"].GetStringSafe();
            auto port = obj["port"].GetUInteger();
            Y_ENSURE(port <= Max<ui16>(), "Port value is out of range");
            host.Port = port;

            return host;
        }
    };

    struct TGencfgGroup {
        TString Tag;
        TString GroupName;
    };

    class TNannyResolver: public IHostGroupResolver, protected THttpResolverBase {
    public:
        TNannyResolver(TNannyResolverConfig conf, TMetricRegistry& registry)
            : THttpResolverBase{conf}
            , Config_{std::move(conf)}
            , Name_{conf.ClusterConfig.Env == ENannyEnv::PRODUCTION ?
                TStringBuilder() << "nanny:" << Config_.ClusterConfig.Service
                : TStringBuilder() << "adm-nanny:" << Config_.ClusterConfig.Service}
            , NannyCounters_{"nanny", registry}
        {
        }

        TAsyncResolveResult Resolve() noexcept override {
            // If cfg groups are set, then we have to first obtain their definitions from nanny
            // and then resolve them to host list via gencfg API
            TIntrusivePtr<TNannyResolver> self{this};

            return DoRequest(BuildRequest(TStringBuf("/current_state/instances/")), [self] (const TString& response) {
                return self->HandleInstances(response);
            }, NannyCounters_);
        }

        const TString& Name() const override {
            return Name_;
        }

        TResolveResult HandleInstances(TStringBuf response) noexcept try {
            auto&& nannyHosts = ParseInstancesResponse(response);
            TUrls result;
            result.reserve(nannyHosts.size());
            for (auto&& host: nannyHosts) {
                result.push_back(FromNannyHost(host, Config_));
            }

            return TResolveResult::FromValue(result);
        } catch (...) {
            return TResolveError{TStringBuilder() << "Failed to parse response: " << CurrentExceptionMessage()};
        }

    protected:
        static TVector<TNannyHost> ParseInstancesResponse(TStringBuf raw) {
            NJson::TJsonValue obj;
            NJson::ReadJsonTree(raw, &obj, true);

            auto&& result = obj["result"].GetArraySafe();

            TVector<TNannyHost> hosts;
            for (auto&& record: result) {
                hosts.push_back(TNannyHost::FromJson(record));
            }

            return hosts;
        }

        static THostAndLabels FromNannyHost(const TNannyHost& nannyHost, const TNannyResolverConfig& config) {
            auto&& cluster = config.ClusterConfig;

            auto&& host = nannyHost.NetworkSettings == ENetworkSettings::NoIsolation
                ? nannyHost.Hostname
                : nannyHost.ContainerHostname;

            THostAndLabels hostAndLabels{host, config.ClusterConfig.Labels};
            hostAndLabels.PortShift = cluster.PortShift;

            if (cluster.UseFetchedPort) {
                hostAndLabels.Port = nannyHost.Port;
            }

            return hostAndLabels;
        }

        IRequestPtr BuildRequest(TStringBuf path) const {
            if (Config_.ClusterConfig.Env == ENannyEnv::ADMIN) {
                TString url = TStringBuilder() << ADMIN_API_URL << Config_.ClusterConfig.Service << path;
                return Get(std::move(url), Headers(HEADERS));
            } else {
                TString url = TStringBuilder() << API_URL << Config_.ClusterConfig.Service << path;
                return Get(std::move(url), Headers(HEADERS));
            }
        }

    protected:
        TNannyResolverConfig Config_;
        TString Name_;
        TCounters NannyCounters_;
    };

    class TGencfgResolver: public TNannyResolver{
        static TString MakeName(const TNannyResolverConfig& config) {
            TStringBuilder sb;
            sb << "gencfg:" << config.ClusterConfig.Service << '/';

            auto& groups = config.ClusterConfig.CfgGroups;
            for (auto i = 0u; i < groups.size(); ++i) {
                auto& group = groups[i];
                sb << group;
                if (i < groups.size() - 1) {
                    sb << ',';
                }
            }

            return sb;
        }

    public:
        TGencfgResolver(TNannyResolverConfig conf, TMetricRegistry& registry)
            : TNannyResolver{conf, registry}
            , Config_{std::move(conf)}
            , GencfgCounters_{"gencfg", registry}
        {
            Name_ = MakeName(Config_);
        }

        TAsyncResolveResult Resolve() noexcept override {
            auto ctx = MakeIntrusive<TWaitContext>();
            TIntrusivePtr self{this};

            DoRequest(BuildRequest(TStringBuf("/runtime_attrs/instances/")), [=] (const TString& response) mutable {
                auto r = ParseGroupsResponse(response);
                if (r.Fail()) {
                    auto e = r.Error();
                    ctx->Fail(std::move(e));
                    return r.PassError<void>();
                }

                auto&& groups = r.Extract();
                for (auto group: groups) {
                    DoRequest(BuildGencfgRequest(group), [=] (const TString& raw) mutable {
                        return ParseGenCfgResponse(raw, self->Config_);
                    }, self->GencfgCounters_).Subscribe([ctx] (auto&& f) {
                        ctx->Complete(f);
                    });
                }

                return TErrorOr<void, TResolveError>::FromValue();
            }, NannyCounters_).Subscribe([ctx] (auto f) {
                auto r = f.ExtractValue();
                if (r.Fail()) {
                    ctx->Fail(r.ExtractError());
                }
            });

            return ctx->Promise.GetFuture();
        }

        static TErrorOr<TVector<TGencfgGroup>, TResolveError> ParseGroupsResponse(TStringBuf raw) noexcept try {
            NJson::TJsonValue obj;
            NJson::ReadJsonTree(raw, &obj, true);

            auto&& groups = obj["content"]["extended_gencfg_groups"]["groups"].GetArraySafe();

            TVector<TGencfgGroup> result;
            for (auto&& group: groups) {
                auto& o = result.emplace_back();
                o.GroupName = group["name"].GetStringSafe();
                o.Tag = group["release"].GetStringSafe();
            }

            return {std::move(result)};
        } catch (...) {
            return TResolveError{TString{"Failed to parse response: "} + CurrentExceptionMessage()};
        }


    private:
        IRequestPtr BuildGencfgRequest(const TGencfgGroup& group) const {
            TString url = TStringBuilder() << GENCFG_API_URL << group.Tag << "/groups/" << group.GroupName << "/instances";
            return Get(std::move(url));
        }

        static TResolveResult ParseGenCfgResponse(TStringBuf raw, const TNannyResolverConfig& config) noexcept try {
            NJson::TJsonValue obj;
            NJson::ReadJsonTree(raw, &obj, true);

            auto&& cluster = config.ClusterConfig;

            auto&& instances = obj["instances"].GetArraySafe();

            TUrls result;
            for (auto&& host: instances) {
                auto& hostAndLabels = result.emplace_back(
                    host["hostname"].GetStringSafe(),
                    config.ClusterConfig.Labels);

                if (config.ClusterConfig.UseFetchedPort) {
                    hostAndLabels.Port = host["port"].GetUIntegerSafe();
                }

                hostAndLabels.PortShift = cluster.PortShift;
            }

            return result;
        } catch (...) {
            return TResolveError{TStringBuilder() << "Failed to parse response: " << CurrentExceptionMessage()};
        }


    private:
        TNannyResolverConfig Config_;
        TCounters GencfgCounters_;
    };
} // namespace

IHostGroupResolverPtr CreateNannyResolver(TNannyResolverConfig config, TMetricRegistry& registry) {
    if (config.ClusterConfig.CfgGroups.empty()) {
        return new TNannyResolver{std::move(config), registry};
    } else {
        return new TGencfgResolver{std::move(config), registry};
    }
}

} // namespace NSolomon::NFetcher
