#include <solomon/services/fetcher/lib/host_groups/config_parser.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <algorithm>

using namespace NSolomon::NFetcher;
using namespace testing;

namespace NSolomon::NFetcher {

    bool operator==(const TConductorConfig& lhs, const TConductorConfig& rhs) {
        return lhs.Name == rhs.Name && lhs.Labels == rhs.Labels;
    }

    bool operator==(const THostPatternConfig& lhs, const THostPatternConfig& rhs) {
        return lhs.Pattern == rhs.Pattern && lhs.Labels == rhs.Labels && lhs.Ranges == rhs.Ranges && lhs.Dc == rhs.Dc;
    }

    bool operator==(const THostListUrlConfig& lhs, const THostListUrlConfig& rhs) {
        return lhs.Url == rhs.Url && lhs.Labels == rhs.Labels && lhs.IgnorePorts == rhs.IgnorePorts;
    }

    bool operator==(const TQloudConfig& lhs, const TQloudConfig& rhs) {
        return lhs.Labels == rhs.Labels && lhs.Application == rhs.Application
            && lhs.Environment == rhs.Environment && lhs.Component == rhs.Component
            && lhs.Project == rhs.Project && lhs.Deployment == rhs.Deployment;
    }

    bool operator==(const TNannyConfig& lhs, const TNannyConfig& rhs) {
        return lhs.Service == rhs.Service && lhs.Labels == rhs.Labels && lhs.Env == rhs.Env
            && lhs.UseFetchedPort == rhs.UseFetchedPort && lhs.PortShift == rhs.PortShift
            && Equal(lhs.CfgGroups.begin(), lhs.CfgGroups.end(), rhs.CfgGroups.begin(), rhs.CfgGroups.end());
    }

    bool operator==(const TInstanceGroupConfig& lhs, const TInstanceGroupConfig& rhs) {
        return lhs.GroupId == rhs.GroupId && lhs.FolderId == rhs.FolderId && lhs.Labels == rhs.Labels;
    }

    std::ostream& operator<<(std::ostream& os, const TNannyConfig& conf) {
        os << "service: ";
        os.write(conf.Service.begin(), conf.Service.size());
        os << "\nportShift: " << conf.PortShift << "\n";
        TString labelsString = TStringBuilder() << "labels: " << conf.Labels << "\n";
        os.write(labelsString.begin(), labelsString.size());
        os << "cfg groups: [";

        for (auto group: conf.CfgGroups) {
            os.write(group.begin(), group.size());
            os << ",";
        }

        os << "]\n";

        return os;
    }
}

TEST(TConductorParserTest, ConductorGroupParsing) {
    const TString data{R"(
        [{"group": "group1", "labels" : ["type=test"]}, {"group": "group2", "labels": ["type=prod"]},
         {"group": "group3", "labels" : []}
        ]
    )"};

    auto groups = ParseConductorGroups(data);

    TVector<TConductorConfig> expected{
        TConductorConfig{"group1", {{"type", "test"}}},
        TConductorConfig{"group2", {{"type", "prod"}}},
        TConductorConfig{"group3", {}},
    };

    ASSERT_THAT(groups, UnorderedElementsAreArray(expected));
}

TEST(TConductorParserTest, EmptyConductorGroup) {
    const TString emptyData{"[]"};

    auto groups = ParseConductorGroups(emptyData);
    ASSERT_THAT(groups, IsEmpty());
}

TEST(TConductorParserTest, ThrowsOnInvalidFormat) {
    {
        const TString data{R"([{"group":"somtething"})"};

        ASSERT_THROW(ParseConductorGroups(data), yexception);
        ASSERT_THROW(ParseConductorTags(data), yexception);
    }

    {
        const TVector<TString> invalidLabel{
            R"([{"name": "group1", "labels" : ["test"]})",
            R"({"name": "group2", "labels": ["type="]}])",
            R"({"name": "group2", "labels": [{"type":"bar"}])"};

        for (auto&& data: invalidLabel) {
            ASSERT_THROW(ParseConductorGroups(data), yexception);
            ASSERT_THROW(ParseConductorTags(data), yexception);
        }
    }
}

TEST(TConductorParserTest, ConductorTagParsing) {
    const TString data{R"(
        [{"name": "group1", "labels" : ["type=test"]}, {"name": "group2", "labels": ["type=prod"]}]
    )"};

    auto groups = ParseConductorTags(data);

    TVector<TConductorConfig> expected{
        TConductorConfig{"group1", {{"type", "test"}}},
        TConductorConfig{"group2", {{"type", "prod"}}},
    };

    ASSERT_THAT(groups, UnorderedElementsAreArray(expected));
}

TEST(TConductorParserTest, EmptyConductorTag) {
    const TString emptyData{"[]"};

    auto groups = ParseConductorGroups(emptyData);
    ASSERT_THAT(groups, IsEmpty());
}

TEST(THostPatternConfigTest, MultipleHosts) {
    const TString data{R"(
        [{"urlPattern":"distbuild00d.search.yandex.net","ranges":"","dc":"","labels":[]},{"urlPattern":"distbuild01d.search.yandex.net","ranges":"","dc":"","labels":[]}]
    )"};

    auto confs = ParseHostPatternConfig(data);
    const TVector<THostPatternConfig> expected{
        THostPatternConfig{"distbuild00d.search.yandex.net", ""},
        THostPatternConfig{"distbuild01d.search.yandex.net", ""},
    };

    ASSERT_THAT(confs, UnorderedElementsAreArray(expected));
}

TEST(THostPatternConfigTest, SingleWithLabels) {
    const TString data{R"(
        [{"urlPattern":"pylontest-sas-%02d.aab.yandex.net","ranges":"01","dc":"sas","labels":["env=test","dc=sas"]}]
    )"};

    auto confs = ParseHostPatternConfig(data);
    const TVector<THostPatternConfig> expected{
        THostPatternConfig{"pylontest-sas-%02d.aab.yandex.net", "01", {{"env", "test"}, {"dc", "sas"}}, "sas"},
    };

    ASSERT_THAT(confs, UnorderedElementsAreArray(expected));
}

TEST(THostListUrlConfigTest, Simple) {
    const TString data{R"(
        [{"url":"https://axis-api-tst.qloud.yandex-team.ru/factprocessor/m/solomon/hosts","labels":[],"ignorePorts":false}]
    )"};

    auto confs = ParseHostUrlConfig(data);
    const TVector<THostListUrlConfig> expected{
        THostListUrlConfig{"https://axis-api-tst.qloud.yandex-team.ru/factprocessor/m/solomon/hosts"}
    };

    ASSERT_THAT(confs, UnorderedElementsAreArray(expected));
}

TEST(TQloudConfigTest, Simple) {
    const TString data{R"(
        [{"component":"app","environment":"production","application":"my-app","project":"my-proj","deployment":"stable","labels":[]}]
    )"};

    auto confs = ParseQloudConfig(data);
    TQloudConfig c;
    c.Component = "app";
    c.Environment = "production";
    c.Application = "my-app";
    c.Project = "my-proj";
    c.Deployment = "stable";

    const TVector<TQloudConfig> expected{c};

    ASSERT_THAT(confs, UnorderedElementsAreArray(expected));
}

TEST(TQloudConfigTest, BadData) {
    const TString data{R"(
        "{"component":"app","environment":"production","application":"my-app","project":"my-proj","deployment":"stable","labels":[]}"
    )"};

    ASSERT_THROW(ParseQloudConfig(data), yexception);
}

TEST(TNannyConfigTest, Simple) {
    const TString data{R"(
        [{"service":"my-service","labels":["foo=bar"],"useFetchedPort":true,"env":"production","portShift":42,"cfgGroup":[]}]
    )"};

    auto confs = ParseNannyConfig(data);
    TNannyConfig c;
    c.Service = "my-service";
    c.PortShift = 42;
    c.UseFetchedPort = true;
    c.Env = ENannyEnv::PRODUCTION;
    c.Labels = {{"foo", "bar"}};

    const TVector<TNannyConfig> expected{c};

    ASSERT_THAT(confs, UnorderedElementsAreArray(expected));
}

TEST(TNannyConfigTest, CfgGroups) {
    const TString data{R"(
        [{"service":"my-service","labels":["foo=bar"],"useFetchedPort":false,"env":"production","portShift":0,"cfgGroup":["MAN_MY_SERVICE","VLA_MY_SERVICE"]}]
    )"};

    auto confs = ParseNannyConfig(data);
    TNannyConfig c;
    c.Service = "my-service";
    c.PortShift = 0;
    c.UseFetchedPort = false;
    c.Labels = {{"foo", "bar"}};
    c.CfgGroups = {"MAN_MY_SERVICE", "VLA_MY_SERVICE"};

    const TVector<TNannyConfig> expected{c};

    ASSERT_THAT(confs, UnorderedElementsAreArray(expected));
}

TEST(TNannyConfigTest, AdminNanny) {
    const TString data{R"(
        [{"service":"my-service","labels":["foo=bar"],"useFetchedPort":true,"env":"admin","portShift":42,"cfgGroup":[]}]
    )"};

    auto confs = ParseNannyConfig(data);
    TNannyConfig c;
    c.Service = "my-service";
    c.PortShift = 42;
    c.UseFetchedPort = true;
    c.Env = ENannyEnv::ADMIN;
    c.Labels = {{"foo", "bar"}};

    const TVector<TNannyConfig> expected{c};

    ASSERT_THAT(confs, UnorderedElementsAreArray(expected));
}

TEST(TNannyConfigTest, PortShitOutOfRange) {
    const TString data{R"(
        [{"service":"my-service","labels":["foo=bar"],"useFetchedPort":false,"portShift":100000,"cfgGroup":[]}]
    )"};

    ASSERT_THROW(ParseNannyConfig(data), yexception);
}

TEST(TNannyConfigTest, YtConfig) {
    auto data = R"(
    [
      {
        "cfgGroup": [
          "VLA_YT_LANDAU_CONTROLLER_AGENTS"
        ],
        "portShift": 0,
        "useFetchedPort": false,
        "env": "PRODUCTION",
        "labels": null,
        "service": "vla_yt_landau_controller_agents"
      },
      {
        "cfgGroup": [
          "VLA_YT_LANDAU_MASTERS"
        ],
        "portShift": 0,
        "useFetchedPort": false,
        "env": "PRODUCTION",
        "labels": null,
        "service": "vla_yt_landau_masters"
      },
      {
        "cfgGroup": [
          "VLA_YT_LANDAU_DEFAULT_TABLET_NODES",
          "VLA_YT_LANDAU_DEFAULT_TABLET_NODES_WITH_XURMA_JOURNALS",
          "VLA_YT_LANDAU_DEFAULT_TABLET_NODES_WITH_XURMA_BLOBS"
        ],
        "portShift": 0,
        "useFetchedPort": false,
        "env": "ADMIN",
        "labels": null,
        "service": "vla_yt_landau_nodes"
      },
      {
        "cfgGroup": [
          "VLA_YT_LANDAU_DEFAULT_DATA_PROXIES"
        ],
        "portShift": 0,
        "useFetchedPort": false,
        "labels": null,
        "service": "vla_yt_landau_proxies"
      },
      {
        "cfgGroup": [
          "VLA_YT_LANDAU_RPC_PROXIES"
        ],
        "portShift": 0,
        "useFetchedPort": false,
        "labels": null,
        "service": "vla_yt_landau_rpc_proxies"
      },
      {
        "cfgGroup": [
          "VLA_YT_LANDAU_SCHEDULERS"
        ],
        "portShift": 0,
        "useFetchedPort": false,
        "env": "production",
        "labels": null,
        "service": "vla_yt_landau_schedulers"
      }
    ]
    )";

    ParseNannyConfig(data);
}

TEST(TYpConfigTest, ParsePodSet) {
    const TString data{R"(
        [{"cluster":"sas-test","podSetId":"foo","labels":["name1=value1", "name2=value2"]}]
    )"};

    auto confs = ParseYpConfig(data);

    TVector<TYpConfig> expected {
        { .Type = TYpPodSet{.Id= "foo"}, .Cluster = "sas-test", .Labels = {{"name1", "value1"}, {"name2", "value2"}} }
    };

    ASSERT_THAT(confs, UnorderedElementsAreArray(expected));
}
TEST(TYpConfigTest, ParseLabel) {
    const TString data{R"(
        [{"cluster": "sas-test", "ypLabel":"/some_label", "tvmLabel":"/labels/foo/bar","labels":["name1=value1", "name2=value2"]}]
    )"};

    auto confs = ParseYpConfig(data);

    TVector<TYpConfig> expected {
        { .Type = TYpLabel{.Value="/some_label", .TvmLabel = "/labels/foo/bar"}, .Cluster = "sas-test", .Labels = {{"name1", "value1"}, {"name2", "value2"}}}
    };

    ASSERT_THAT(confs, UnorderedElementsAreArray(expected));
}
TEST(TYpConfigTest, ParseEndpointSet) {
    const TString data{R"(
        [{"cluster":"sas-test","endpointSetId":"foo","labels":["name1=value1", "name2=value2"]}]
    )"};

    auto confs = ParseYpConfig(data);

    TVector<TYpConfig> expected {
        { .Type = TYpEndpointSet{.Id="foo"}, .Cluster = "sas-test", .Labels = {{"name1", "value1"}, {"name2", "value2"}}}
    };

    ASSERT_THAT(confs, UnorderedElementsAreArray(expected));
}

TEST(TYpConfigTest, NullsAreValid) {
    const TString data{R"(
        [{"podSetId":"adfox-matchid-production.matchid","cluster":"sas","labels":[],"tvmLabel":"","ypLabel":""},{"podSetId":"adfox-matchid-production.matchid","cluster":"vla","labels":[],"tvmLabel":"","ypLabel":""},{"podSetId":"adfox-matchid-production.matchid","cluster":"myt","labels":[],"tvmLabel":null,"ypLabel":null}]
    )"};

    auto confs = ParseYpConfig(data);
    ASSERT_THAT(confs, SizeIs(3));
    for (auto conf: confs) {
        ASSERT_TRUE(std::holds_alternative<TYpPodSet>(conf.Type)) << "wrong type in std::variant";
        ASSERT_EQ(std::get<TYpPodSet>(conf.Type).Id, "adfox-matchid-production.matchid");
    }

    ASSERT_THAT(confs, Each(Field(&TYpConfig::Labels, IsEmpty())));
}

TEST(TYpConfigTest, NullsAreValidEndpoints) {
    const TString data{R"(
        [{"endpointSetId":"adfox-matchid-production.matchid","cluster":"sas","labels":[]},{"endpointSetId":"adfox-matchid-production.matchid","cluster":"vla","labels":[]},{"endpointSetId":"adfox-matchid-production.matchid","cluster":"myt","labels":[]}]
    )"};

    auto confs = ParseYpConfig(data);
    ASSERT_THAT(confs, SizeIs(3));
    for (auto conf: confs) {
        ASSERT_TRUE(std::holds_alternative<TYpEndpointSet>(conf.Type)) << "wrong type in std::variant";
        ASSERT_EQ(std::get<TYpEndpointSet>(conf.Type).Id, "adfox-matchid-production.matchid");
    }

    ASSERT_THAT(confs, Each(Field(&TYpConfig::Labels, IsEmpty())));
}

TEST(TYpConfigTest, PodSetIsEmptyString) {
    const TString data{R"(
        [{"podSetId":"","cluster":"sas","labels":[],"tvmLabel":"","ypLabel":"/some_label"},{"podSetId":"","cluster":"vla","labels":[],"tvmLabel":"","ypLabel":"/some_label"},{"podSetId":"","cluster":"myt","labels":[],"tvmLabel":null,"ypLabel":"/some_label"}]
    )"};

    auto confs = ParseYpConfig(data);
    ASSERT_THAT(confs, SizeIs(3));
    for (auto conf: confs) {
        ASSERT_TRUE(std::holds_alternative<TYpLabel>(conf.Type)) << "wrong type in std::variant";
        ASSERT_EQ(std::get<TYpLabel>(conf.Type).Value, "/some_label");
    }

    ASSERT_THAT(confs, Each(Field(&TYpConfig::Labels, IsEmpty())));
}

TEST(TYpConfigTest, EmptyStringsAreValid) {
    const TString data{R"(
        [{"podSetId":"foo","endpointSetId":"","cluster":"sas","labels":[],"tvmLabel":"","ypLabel":""},{"endpodSetId":"","podSetId":"","cluster":"vla","labels":[],"tvmLabel":"/labels/foo/bar","ypLabel":"/some_label"},{"posSetId":"","endpointSetId":"foo","cluster":"myt","labels":[],"tvmLabel":null,"ypLabel":""}]
    )"};

    auto confs = ParseYpConfig(data);
    ASSERT_THAT(confs, SizeIs(3));

    ASSERT_TRUE(std::holds_alternative<TYpPodSet>(confs[0].Type)) << "wrong type in std::variant";
    ASSERT_EQ(std::get<TYpPodSet>(confs[0].Type).Id, "foo");

    ASSERT_TRUE(std::holds_alternative<TYpLabel>(confs[1].Type)) << "wrong type in std::variant";
    ASSERT_EQ(std::get<TYpLabel>(confs[1].Type).Value, "/some_label");

    ASSERT_TRUE(std::holds_alternative<TYpEndpointSet>(confs[2].Type)) << "wrong type in std::variant";
    ASSERT_EQ(std::get<TYpEndpointSet>(confs[2].Type).Id, "foo");

    ASSERT_THAT(confs, Each(Field(&TYpConfig::Labels, IsEmpty())));
}

TEST(TInstanceGroupConfigTest, Simple) {
    const TString data{R"(
        [{"instanceGroupId":"foo","labels":["name1=value1", "name2=value2"]}]
    )"};

    auto confs = ParseInstanceGroupConfig(data);

    TVector<TInstanceGroupConfig> expected {
        { .GroupId = "foo", .Labels = {{"name1", "value1"}, {"name2", "value2"}} }
    };

    ASSERT_THAT(confs, UnorderedElementsAreArray(expected));
}

TEST(TInstanceGroupConfigTest, Folder) {
    const TString data{R"(
        [{"folderId":"foo","labels":["name1=value1", "name2=value2"]}]
    )"};

    auto confs = ParseInstanceGroupConfig(data);

    TVector<TInstanceGroupConfig> expected {
        { .FolderId = "foo", .Labels = {{"name1", "value1"}, {"name2", "value2"}} }
    };

    ASSERT_THAT(confs, UnorderedElementsAreArray(expected));
}

TEST(TInstanceGroupConfigTest, BrokenFormat) {
    {
        const TString data{R"(
            [{"instanceGroupId":"foo","labels":""}]
        )"};

        ASSERT_THROW(ParseInstanceGroupConfig(data), yexception);
    }

    {
        const TString data{R"(
            [{"instanceGroupId":[],"labels":[]}]
        )"};

        ASSERT_THROW(ParseInstanceGroupConfig(data), yexception);
    }

    {
        const TString data{R"(
            [{"id":"", labels":[]}]
        )"};

        ASSERT_THROW(ParseInstanceGroupConfig(data), yexception);
    }

    {
        const TString data{R"(
            [{"instanceGroupId":"ig","folderId":"folder","labels":[]}]
        )"};

        ASSERT_THROW(ParseInstanceGroupConfig(data), yexception);
    }
}

TEST(TCloudDnsConfigTest, Simple) {
    const TString data{R"(
        [{"env":"PROD","labels":["name1=value1", "name2=value2"], "name": "some-dns-name"}]
    )"};

    auto confs = ParseCloudDnsConfig(data);

    TVector<TCloudDnsConfig> expected {
        { .Env = ECloudEnv::PROD, .Name = "some-dns-name", .Labels = {{"name1", "value1"}, {"name2", "value2"}} }
    };

    ASSERT_THAT(confs, UnorderedElementsAreArray(expected));
}

TEST(TCloudDnsConfigTest, InvalidEnv) {
    const TString data{R"(
        [{"env":"ABRACADABRA","labels":["name1=value1", "name2=value2"], "name": "some-dns-name"}]
    )"};

    auto confs = ParseCloudDnsConfig(data);

    TVector<TCloudDnsConfig> expected {
        { .Env = ECloudEnv::UNKNOWN, .Name = "some-dns-name", .Labels = {{"name1", "value1"}, {"name2", "value2"}} }
    };

    ASSERT_THAT(confs, UnorderedElementsAreArray(expected));
}

TEST(TCloudDnsConfigTest, InvalidName) {
    {
        const TString data{R"(
            [{"env":"PREPROD","labels":["name1=value1", "name2=value2"]}]
        )"};

        ASSERT_THROW(ParseCloudDnsConfig(data), yexception);
    }
    {
        const TString data{R"(
            [{"env":"PREPROD","labels":["name1=value1", "name2=value2"], "name": ""}]
        )"};

        ASSERT_THROW(ParseCloudDnsConfig(data), yexception);
    }
}
