#include <solomon/services/fetcher/lib/host_groups/host_and_labels.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NMonitoring;
using namespace NSolomon::NFetcher;
using namespace testing;

TEST(THostAndLabelsTest, FromHostname) {
    auto data = R"(foo.yandex.net)";

    auto result = THostAndLabels::FromString(data);

    ASSERT_THAT(result.Host, Eq(data));
    ASSERT_THAT(result.IpAddress, Eq(Nothing()));
    ASSERT_THAT(result.Port, Eq(THostAndLabels::DEFAULT_PORT));
};

TEST(THostAndLabelsTest, FromIpv4Address) {
    auto data = R"(127.0.0.1)";

    auto result = THostAndLabels::FromString(data);

    ASSERT_THAT(result.Host, IsEmpty());
    ASSERT_TRUE(result.IpAddress.Defined());
    auto& ipAddress = result.IpAddress.GetRef();
    ASSERT_THAT(ipAddress.Type(), Eq(TIpv6Address::Ipv4));
    ASSERT_THAT(ipAddress, Eq(::Get127001()));
    ASSERT_THAT(result.Port, Eq(THostAndLabels::DEFAULT_PORT));
};

TEST(THostAndLabelsTest, FromIpv6AddressWithPort) {
    auto data = R"([2a02:6b8:b000:a308:215:b2ff:fea9:67d2]:81)";

    auto result = THostAndLabels::FromString(data);

    ASSERT_THAT(result.Host, IsEmpty());
    ASSERT_TRUE(result.IpAddress.GetRef().IsValid());
    ASSERT_THAT(result.Port, Eq(81));
};

TEST(THostAndLabelsTest, FromIpv6AddressWithPortAndLabels) {
    {
        auto data = R"([2a02:6b8:b000:a308:215:b2ff:fea9:67d2]:81 foo=bar)";

        auto result = THostAndLabels::FromString(data);
        ASSERT_THAT(result.Labels, Eq(TLabels{{"foo", "bar"}}));
    }

    {
        auto data = R"([2a02:6b8:b000:a308:215:b2ff:fea9:67d2]:81 foo=bar baz=fiz)";

        auto result = THostAndLabels::FromString(data);
        ASSERT_THAT(result.Labels, Eq(TLabels{{"foo", "bar"}, {"baz", "fiz"}}));
    }
};

TEST(THostAndLabelsTest, FromEmptyString) {
    ASSERT_THROW(THostAndLabels::FromString(""), yexception);
}

TEST(THostAndLabelsTest, FromStringWithTrailingSpace) {
    auto result = THostAndLabels::FromString("foo.bar ");
    ASSERT_THAT(result.Host, Eq("foo.bar"));
    ASSERT_THAT(result.Labels, Eq(TLabels{}));
}

TEST(THostAndLabelsTest, WithPortShift) {
    {
        auto data = R"([2a02:6b8:b000:a308:215:b2ff:fea9:67d2]:81s1 foo=bar)";

        auto result = THostAndLabels::FromString(data);
        ASSERT_THAT(result.Port, Eq(81));
        ASSERT_THAT(result.PortShift, Eq(1));
    }
    {
        auto data = R"([2a02:6b8:b000:a308:215:b2ff:fea9:67d2]:s1 foo=bar)";

        auto result = THostAndLabels::FromString(data);
        ASSERT_THAT(result.Port, Eq(THostAndLabels::DEFAULT_PORT));
        ASSERT_THAT(result.PortShift, Eq(1));
    }

    {
        auto data = R"([2a02:6b8:b000:a308:215:b2ff:fea9:67d2]:s-1 foo=bar)";

        auto result = THostAndLabels::FromString(data);
        ASSERT_THAT(result.Port, Eq(THostAndLabels::DEFAULT_PORT));
        ASSERT_THAT(result.PortShift, Eq(-1));
    }
}

TEST(THostAndLabelsTest, WithIncorrectPortShift) {
    {
        auto data = R"([2a02:6b8:b000:a308:215:b2ff:fea9:67d2]:81ss1 foo=bar)";
        ASSERT_THROW(THostAndLabels::FromString(data), yexception);
    }
    {
        auto data = R"([2a02:6b8:b000:a308:215:b2ff:fea9:67d2]:81s1s5 foo=bar)";
        ASSERT_THROW(THostAndLabels::FromString(data), yexception);
    }
}

TEST(THostAndLabelsTest, Serialization) {
    {
        auto data = R"([2a02:6b8:b000:a308:215:b2ff:fea9:67d2]:81 foo=bar)";
        auto result = THostAndLabels::FromString(data);
        ASSERT_THAT(result.ToString(), Eq(data));
    }
    {
        auto data = R"([2a02:6b8:b000:a308:215:b2ff:fea9:67d2]:81s1 foo=bar)";
        auto result = THostAndLabels::FromString(data);
        ASSERT_THAT(result.ToString(), Eq(data));
    }
    {
        auto data = R"([2a02:6b8:b000:a308:215:b2ff:fea9:67d2]:s3 foo=bar)";
        auto result = THostAndLabels::FromString(data);
        ASSERT_THAT(result.ToString(), Eq(data));
    }
    {
        auto data = R"(foo.yandex.net:s3 foo=bar)";
        auto result = THostAndLabels::FromString(data);
        ASSERT_THAT(result.ToString(), Eq(data));
    }
    {
        auto data = R"(127.0.0.1:s3 foo=bar fiz=baz)";
        auto result = THostAndLabels::FromString(data);
        ASSERT_THAT(result.ToString(), Eq(data));
    }
}
