#include <solomon/services/fetcher/lib/host_groups/host_resolver.h>
#include <solomon/services/fetcher/lib/host_groups/yd.h>
#include <solomon/services/fetcher/lib/host_groups/yp.h>

#include <solomon/libs/cpp/http/client/curl/client.h>
#include <solomon/services/fetcher/testlib/http_server.h>

#include <infra/yp_service_discovery/api/api.pb.h>

#include <library/cpp/http/misc/httpcodes.h>
#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/resource/resource.h>
#include <library/cpp/testing/gtest/gtest.h>

#include <util/string/split.h>

using namespace NSolomon;
using namespace NMonitoring;
using namespace NSolomon::NFetcher;
using namespace NSolomon::NTesting;
using namespace testing;


namespace NSolomon::NFetcher {
    // allow gtest to print pretty values instead of bytes
    std::ostream& operator<<(std::ostream& os, const THostAndLabels& h) {
        auto s = h.ToString();
        os.write(s.begin(), s.size());

        return os;
    }
}

const TString TAG_RESPONSE{R"(golem1-iva.yandex.net
golem1-sas.yandex.net
golem1.myt.yandex.net
golem2-iva.yandex.net
golem2-sas.yandex.net
golem2.myt.yandex.net
golem3-sas.yandex.net)"};

const TString GROUP_RESPONSE{R"(solomon-pre-front-sas-00.search.yandex.net
solomon-pre-front-sas-01.search.yandex.net
solomon-pre-front-sas-02.search.yandex.net
solomon-pre-alert-00.search.yandex.net
solomon-pre-alert-01.search.yandex.net
solomon-pre-alert-02.search.yandex.net)"};

TVector<THostAndLabels> GetValue(TAsyncResolveResult&& r) {
    return r.ExtractValueSync().Extract();
}

TResolveError GetError(TAsyncResolveResult&& r) {
    return r.ExtractValueSync().ExtractError();
}

class TConductorServer: public TTestServer {
public:
    TConductorServer() {
        AddHandler("/api/groups2hosts/solomon_pre", [] {
            THttpResponse resp;
            resp.SetContent(GROUP_RESPONSE);
            return resp;
        });

        AddHandler("/api/tag2hosts/golem", [] {
            THttpResponse resp;
            resp.SetContent(TAG_RESPONSE);
            return resp;
        });
    }
};

const TString HOST_LIST_RESPONSE = R"(
    my1.yandex.net
    my2.yandex.net
    my3.yandex.net
)";

const TUrls HOST_LIST_HOSTS = {
    THostAndLabels::FromString("my1.yandex.net"),
    THostAndLabels::FromString("my2.yandex.net"),
    THostAndLabels::FromString("my3.yandex.net"),
};

class THostListServer: public TTestServer {
public:
    THostListServer() {
        AddHandler("/", [] {
            THttpResponse resp;
            resp.SetContent(HOST_LIST_RESPONSE);
            return resp;
        });
    }
};

TMetricRegistry DUMMY_REGISTRY;
struct: IHttpClient {
    void Request(IRequestPtr, TOnComplete cb, const TRequestOpts&) noexcept override {
        cb(TResult::FromError(TRequestError::EType::Unknown, "Not implemented"));
    }
} DUMMY_CLIENT;

class TConductorTest: public ::testing::Test {
public:
    void SetUp() override {
        HttpClient_.Reset(CreateCurlClient({}, Registry_));
        Srv_.Reset(new TConductorServer);
    }

    TConductorResolverConfig MakeConfig(const TString& name) {
        TConductorResolverConfig result{name, HttpClient_.Get()};
        result.ConductorUrl = Srv_->Address();
        result.Retries = 1;
        return result;
    }

    void TearDown() override {
        Srv_->Stop();
    }

private:
    THolder<TConductorServer> Srv_;
    TMetricRegistry Registry_;
    IHttpClientPtr HttpClient_{CreateCurlClient({}, Registry_)};
};

TEST_F(TConductorTest, ResolvesGroup) {
    auto resolver = CreateConductorGroupResolver(MakeConfig("solomon_pre"), DUMMY_REGISTRY);
    ASSERT_THAT(resolver->Name(), StrEq("conductor-group:solomon_pre"));

    const TUrls urls = GetValue(resolver->Resolve());

    TUrls expectedUrls;
    StringSplitter(GROUP_RESPONSE).Split('\n')
            .Consume([&] (auto line) {
                expectedUrls.push_back(THostAndLabels::FromString(line));
            });

    ASSERT_THAT(urls, UnorderedElementsAreArray(expectedUrls));
}

TEST_F(TConductorTest, ResolvesTag) {
    auto resolver = CreateConductorTagResolver(MakeConfig("golem"), DUMMY_REGISTRY);

    const TUrls urls = GetValue(resolver->Resolve());
    TUrls expectedUrls;
    StringSplitter(TAG_RESPONSE).Split('\n')
            .Consume([&] (auto line) {
                expectedUrls.push_back(THostAndLabels::FromString(line));
            });

    ASSERT_THAT(urls, UnorderedElementsAreArray(expectedUrls));
}

TEST_F(TConductorTest, ThrowsOnNonExisting) {
    auto tagResolver = CreateConductorTagResolver(MakeConfig("foo"), DUMMY_REGISTRY);
    auto groupResolver = CreateConductorGroupResolver(MakeConfig("bar"), DUMMY_REGISTRY);

    ASSERT_THAT(GetError(tagResolver->Resolve()).IsTransient(), false);
    ASSERT_THAT(GetError(groupResolver->Resolve()).IsTransient(), false);
}

const TUrls HOST_RANGE{
    THostAndLabels::FromString("my-0.yandex.net"),
    THostAndLabels::FromString("my-1.yandex.net"),
    THostAndLabels::FromString("my-2.yandex.net"),
};

const TUrls HOST_RANGE_EXTENDED{
    THostAndLabels::FromString("my-5.yandex.net"),
    THostAndLabels::FromString("my-6.yandex.net")
};

TEST(THostListTest, SimpleFormatting) {
    auto resolver = CreateHostPatternResolver(THostPatternResolverConfig{"my-%d.yandex.net", "0-2"});
    auto urls = GetValue(resolver->Resolve());

    ASSERT_THAT(resolver->Name(), StrEq("my-%d.yandex.net"));
    ASSERT_THAT(urls, UnorderedElementsAreArray(HOST_RANGE));

    resolver = CreateHostPatternResolver(THostPatternResolverConfig{"my-%d.yandex.net", "0"});
    urls = GetValue(resolver->Resolve());
}

TEST(THostListTest, ComplexFormatting) {
    static const TUrls expected{
            THostAndLabels::FromString("my-000.yandex.net"),
            THostAndLabels::FromString("my-001.yandex.net"),
            THostAndLabels::FromString("my-002.yandex.net"),
    };

    auto resolver = CreateHostPatternResolver(THostPatternResolverConfig{{"my-%03d.yandex.net"}, {"0-2"}});
    auto urls = GetValue(resolver->Resolve());

    ASSERT_THAT(resolver->Name(), StrEq("my-%03d.yandex.net"));
    ASSERT_THAT(urls, UnorderedElementsAreArray(expected));
}

TEST(THostListTest, SingleHost) {
    auto resolver = CreateHostPatternResolver(THostPatternResolverConfig{"my.yandex.net", {}});
    auto urls = GetValue(resolver->Resolve());
    ASSERT_THAT(urls, ElementsAre(THostAndLabels::FromString("my.yandex.net")));
}

TEST(THostListTest, NoPlaceholder) {
    auto resolver = CreateHostPatternResolver(THostPatternResolverConfig{"my.yandex.net", "0-2"});
    auto urls = GetValue(resolver->Resolve());

    ASSERT_THAT(urls, ElementsAre(THostAndLabels::FromString("my.yandex.net")));
}

TEST(THostListTest, IncorrectRangeFormat) {
    auto resolver = CreateHostPatternResolver(THostPatternResolverConfig{"my.yandex.net", "-2"});
    ASSERT_THAT(GetError(resolver->Resolve()).IsTransient(), false);

    resolver = CreateHostPatternResolver(THostPatternResolverConfig{"my.yandex.net", "-1-2"});
    ASSERT_THAT(GetError(resolver->Resolve()).IsTransient(), false);
}

TEST(THostListTest, IncorrectPatternFormat) {
    auto resolver = CreateHostPatternResolver(THostPatternResolverConfig{"my%s.yandex.net", "1-2"});
    ASSERT_THAT(GetError(resolver->Resolve()).IsTransient(), false);

    resolver = CreateHostPatternResolver(THostPatternResolverConfig{"my%d-%d.yandex.net", "1-2"});
    ASSERT_THAT(GetError(resolver->Resolve()).IsTransient(), false);
}

TEST(THostListTest, MultipleRanges) {
    auto resolver = CreateHostPatternResolver(THostPatternResolverConfig{"my-%d.yandex.net", "0-2 5-6"});
    auto urls = GetValue(resolver->Resolve());

    auto expected = HOST_RANGE;
    Copy(HOST_RANGE_EXTENDED.begin(), HOST_RANGE_EXTENDED.end(), std::back_inserter(expected));
    ASSERT_THAT(urls, UnorderedElementsAreArray(expected));
}

class THostUrlTest: public ::testing::Test {
public:
    void SetUp() override {
        HttpClient_ = CreateCurlClient({}, Registry_);
        Srv_.Reset(new THostListServer);
        Address_ = Srv_->Address();
    }

    void TearDown() override {
        HttpClient_.Reset();
        Srv_.Reset();
    }

protected:
    THolder<THostListServer> Srv_;
    TString Address_;
    TMetricRegistry Registry_;
    IHttpClientPtr HttpClient_{CreateCurlClient({}, Registry_)};
};

TEST_F(THostUrlTest, Simple) {
    auto resolver = CreateHostUrlResolver(THostUrlResolverConfig{Address_, HttpClient_.Get()}, DUMMY_REGISTRY);
    ASSERT_THAT(resolver->Name(), Address_);
    ASSERT_THAT(GetValue(resolver->Resolve()), UnorderedElementsAreArray(HOST_LIST_HOSTS));
}

TEST_F(THostUrlTest, NotFound) {
    const auto url = Address_ + "/hello";
    auto resolver = CreateHostUrlResolver(THostUrlResolverConfig{url, HttpClient_.Get()}, DUMMY_REGISTRY);
    ASSERT_THAT(resolver->Name(), url);
    ASSERT_THAT(GetError(resolver->Resolve()).IsTransient(), false);
}

TEST(TNetworkTest, SingleHost) {
    for (auto netStr: {"10.0.0.1", "fedc:ba09::"}) {
        TNetworkConfig net;
        net.Network = netStr;

        bool ok{};
        THostAndLabels expected{TIpv6Address::FromString(netStr, ok)};
        ASSERT_TRUE(ok);

        auto resolver = CreateNetworkResolver({.ClusterConfig = net});
        ASSERT_THAT(GetValue(resolver->Resolve()), ElementsAre(expected));
        ASSERT_THAT(resolver->Name(), StrEq(TString{"network:"} + netStr));
    }
}

TEST(TNetworkTest, CidrNotation) {
    {
        TNetworkConfig net;
        net.Network = "10.0.0.1/32";

        bool ok{};
        THostAndLabels expected{TIpv6Address::FromString("10.0.0.1", ok)};
        ASSERT_TRUE(ok);

        auto resolver = CreateNetworkResolver({.ClusterConfig = net});
        ASSERT_THAT(resolver->Name(), StrEq(TString{"network:"} + net.Network));
        ASSERT_THAT(GetValue(resolver->Resolve()), ElementsAre(expected));
    }
    {
        TNetworkConfig net;
        net.Network = "fedc:ba09::/126";
        bool ok{};
        THostAndLabels expected1{TIpv6Address::FromString("fedc:ba09::1", ok)};
        ASSERT_TRUE(ok);
        THostAndLabels expected2{TIpv6Address::FromString("fedc:ba09::2", ok)};
        ASSERT_TRUE(ok);

        auto resolver = CreateNetworkResolver({.ClusterConfig = net});
        ASSERT_THAT(GetValue(resolver->Resolve()), ElementsAre(expected1, expected2));
    }
}

TEST(TNannyTest, Naming) {
    TNannyResolverConfig conf{&DUMMY_CLIENT};
    conf.ClusterConfig.Service = "foobar";
    auto resolver = CreateNannyResolver(std::move(conf), DUMMY_REGISTRY);

    ASSERT_THAT(resolver->Name(), "nanny:foobar");
}

TEST(TNannyTest, GencfgNaming) {
    TNannyResolverConfig conf{&DUMMY_CLIENT};
    conf.ClusterConfig.Service = "foobar";
    conf.ClusterConfig.CfgGroups = {"group1", "group2"};

    auto resolver = CreateNannyResolver(std::move(conf), DUMMY_REGISTRY);
    ASSERT_THAT(resolver->Name(), "gencfg:foobar/group1,group2");
}

TEST(TYpTest, Naming) {
    {
        TYpResolverConfig conf{&DUMMY_CLIENT};
        conf.ClusterConfig.Type = TYpLabel{.Value = "foobar"};
        conf.ClusterConfig.Cluster = "sas";
        auto resolver = CreateYpResolver(std::move(conf), DUMMY_REGISTRY);
        ASSERT_THAT(resolver->Name(), "yp-label:sas/foobar");
    }
}

TEST(TYpTest, ParsingPlain) {
    const auto response = NResource::Find("/yp_plain_response.json");
    auto hosts = ParseYpResponse(response, {}).Extract();
    TVector<THostAndLabels> expected {
        THostAndLabels{"ascmh7dfbbn5rbty.man.yp-c.yandex.net", 0, {}},
        THostAndLabels{"pprgtoquuo2wuzze.man.yp-c.yandex.net", 0, {}},
    };

    ASSERT_THAT(hosts, UnorderedElementsAreArray(expected));
}

TEST(TYpTest, ParsingWithTvm) {
    const auto response = NResource::Find("/yp_tvm_response.json");
    auto hosts = ParseYpResponse(response, {}).Extract();
    TVector<THostAndLabels> expected {
        THostAndLabels{"r4xkej4xiiar3gwj.vla.yp-c.yandex.net", 0, {}},
        THostAndLabels{"sgjj67ype2w3aqui.vla.yp-c.yandex.net", 0, {}},
    };

    for (auto& host: expected) {
        host.TvmDestId = 2010882;
    }

    ASSERT_THAT(hosts, UnorderedElementsAreArray(expected));
}

TEST(TYdTest, Naming) {
    {
        TYpResolverConfig conf{&DUMMY_CLIENT};
        conf.ClusterConfig.Type = TYpPodSet{.Id = "foobar"};
        conf.ClusterConfig.Cluster = "sas";
        auto resolver = CreateYdResolver(std::move(conf), DUMMY_REGISTRY);
        ASSERT_THAT(resolver->Name(), "yd-pod:sas/foobar");
    }
    {
        TYpResolverConfig conf{&DUMMY_CLIENT};
        conf.ClusterConfig.Type = TYpEndpointSet{.Id = "foobar"};
        conf.ClusterConfig.Cluster = "sas";
        auto resolver = CreateYdResolver(std::move(conf), DUMMY_REGISTRY);
        ASSERT_THAT(resolver->Name(), "yd-endpoint:sas/foobar");
    }
}

TEST(TYdTest, Parsing) {
    using namespace NYP::NServiceDiscovery;

    NApi::TRspResolvePods resp;
    resp.set_resolve_status(NApi::EResolveStatus::OK);
    {
        NApi::TPodSet* podSet = resp.mutable_pod_set();
        {
            NApi::TPod* pod = podSet->add_pods();
            {
                auto* location = pod->add_ip6_address_allocations();
                location->set_address("2a02:6b8:c1a:31a9:100:0:ffcb:0");
                location->set_vlan_id("backbone");
                location->set_transient_fqdn("man2-9428-1.ascmh7dfbbn5rbty.man.yp-c.yandex.net");
                location->set_persistent_fqdn("ascmh7dfbbn5rbty.man.yp-c.yandex.net");
            }
            {
                auto* location = pod->add_ip6_address_allocations();
                location->set_address("2a02:6b8:fc1b:31a9:100:0:50e2:0");
                location->set_vlan_id("fastbone");
                location->set_transient_fqdn("fb-man2-9428-1.ascmh7dfbbn5rbty.man.yp-c.yandex.net");
                location->set_persistent_fqdn("fb-ascmh7dfbbn5rbty.man.yp-c.yandex.net");
            }
        }
        {
            NApi::TPod* pod = podSet->add_pods();
            {
                auto* location = pod->add_ip6_address_allocations();
                location->set_address("2a02:6b8:c0a:3526:100:0:95f9:0");
                location->set_vlan_id("backbone");
                location->set_transient_fqdn("man2-9292-1.pprgtoquuo2wuzze.man.yp-c.yandex.net");
                location->set_persistent_fqdn("pprgtoquuo2wuzze.man.yp-c.yandex.net");
            }
            {
                auto* location = pod->add_ip6_address_allocations();
                location->set_address("2a02:6b8:fc12:3526:100:0:5936:0");
                location->set_vlan_id("fastbone");
                location->set_transient_fqdn("fb-man2-9292-1.pprgtoquuo2wuzze.man.yp-c.yandex.net");
                location->set_persistent_fqdn("fb-pprgtoquuo2wuzze.man.yp-c.yandex.net");
            }
        }
    }

    auto hosts = ParseYdResponse(resp.SerializeAsStringOrThrow(), {}, {}).Extract();
    TVector<THostAndLabels> expected {
            THostAndLabels{"ascmh7dfbbn5rbty.man.yp-c.yandex.net",0, {}},
            THostAndLabels{"pprgtoquuo2wuzze.man.yp-c.yandex.net", 0, {}},
    };

    ASSERT_THAT(hosts, UnorderedElementsAreArray(expected));
}

TEST(TYpEndpointsTest, Parsing) {
    using namespace NYP::NServiceDiscovery;

    NApi::TRspResolveEndpoints resp;
    resp.set_resolve_status(NApi::EResolveStatus::OK);
    {
        NApi::TEndpointSet* endpointSet = resp.mutable_endpoint_set();
        {
            NApi::TEndpoint* endpoint = endpointSet->add_endpoints();
            {
                endpoint->set_ip6_address("2a02:6b8:c1a:31a9:100:0:ffcb:0");
                endpoint->set_fqdn("ascmh7dfbbn5rbty.man.yp-c.yandex.net");
            }
        }
        {
            NApi::TEndpoint* endpoint = endpointSet->add_endpoints();
            {
                endpoint->set_ip6_address("2a02:6b8:c0a:3526:100:0:95f9:0");
                endpoint->set_fqdn("pprgtoquuo2wuzze.man.yp-c.yandex.net");
            }
        }
        {
            NApi::TEndpoint* endpoint = endpointSet->add_endpoints();
            {
                endpoint->set_ip6_address("2a02:6b8:c14:322b:10d:c689:f047:0%0");
                endpoint->set_fqdn("soy-production-mover-arnold-11.sas.yp-c.yandex.net");
                endpoint->set_port(12000);
            }
        }
    }
    auto hosts = ParseYpEndpointsResponse(resp.SerializeAsString(), {}, {}).Extract();
    TVector<THostAndLabels> expected {
            THostAndLabels{"ascmh7dfbbn5rbty.man.yp-c.yandex.net",0, {}},
            THostAndLabels{"pprgtoquuo2wuzze.man.yp-c.yandex.net", 0, {}},
            THostAndLabels{"soy-production-mover-arnold-11.sas.yp-c.yandex.net", 12000, {}},
    };

    ASSERT_THAT(hosts, UnorderedElementsAreArray(expected));
}
