#include <solomon/services/fetcher/lib/host_groups/agents.h>
#include <solomon/services/fetcher/testlib/http_server.h>

#include <solomon/libs/cpp/http/client/curl/client.h>

#include <library/cpp/cgiparam/cgiparam.h>
#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/resource/resource.h>
#include <library/cpp/testing/gtest/gtest.h>

#include <utility>

using namespace NMonitoring;
using namespace NSolomon;
using namespace NSolomon::NFetcher;
using namespace NSolomon::NTesting;
using namespace testing;

const TString HOSTS_API{"/v1/hosts"};
const TString CONDUCTOR_API{"/api/tag2hosts/yasm_monitored"};

const TString YASM_MONITORED_JSON = NResource::Find("/yasm_monitored.json");
const TString YASM_QLOUD_MONITORED_JSON = NResource::Find("/yasm_qloud_monitored.json");
const TString EMPTY_RESPONSE_JSON = R"({"result": []})";

THttpResponse DefaultHostsHandler(TParsedHttpFull req) {
    THttpResponse result;
    TCgiParameters params{req.Cgi};

    {
        auto it = params.Find("cursor");
        if (it != params.end() && FromString<int>(it->second) != 0) {
            result.SetContent(EMPTY_RESPONSE_JSON);
            return result;
        }
    }

    auto it = params.Find("tags");
    if (it == params.end()) {
        result.SetContent(EMPTY_RESPONSE_JSON);
        return result;
    }

    if (it->second == "yasm_monitored") {
        result.SetContent(YASM_MONITORED_JSON);
    } else if (it->second == "yasm_qloud_monitored") {
        result.SetContent(YASM_QLOUD_MONITORED_JSON);
    } else {
        result.SetContent(EMPTY_RESPONSE_JSON);
    }

    return result;
}


const TVector<TString> YASM_MONITORED_HOSTS = {
    "myt1-6683.search.yandex.net",
    "myt1-6549.search.yandex.net",
    "myt1-6622.search.yandex.net",
    "iva1-5738.search.yandex.net",
    "myt1-6547.search.yandex.net",
    "myt1-6690.search.yandex.net",
    "myt1-6631.search.yandex.net",
    "myt1-6546.search.yandex.net",
    "myt1-6626.search.yandex.net",
    "mrt02e.market.yandex.net",
};

const TVector<TString> YASM_QLOUD_MONITORED_HOSTS = {
    "iva1-1996.search.yandex.net",
    "myt1-1992.search.yandex.net",
    "iva1-1931.search.yandex.net",
    "iva1-7378.search.yandex.net",
    "iva1-0518.search.yandex.net",
    "iva1-7570.search.yandex.net",
    "iva1-0288.search.yandex.net",
    "iva1-7050.search.yandex.net",
    "iva1-0334.search.yandex.net",
    "iva1-0232.search.yandex.net",
};

const TVector<TString> CONDUCTOR_HOSTS = {
    "a34no6zeebnpnuew.vla.yp-c.yandex.net",
    "a3imy5uehnma5aca.vla.yp-c.yandex.net",
    "a5peetyh5jnuqldl.vla.yp-c.yandex.net",
};

THttpResponse DefaultConductorHandler() {
    static const TString RESPONSE = [] {
        TStringBuilder sb;
        for (auto&& h: CONDUCTOR_HOSTS) {
            sb << h << '\n';
        }

        return sb;
    }();

    THttpResponse result;
    result.SetContent(RESPONSE);
    return result;
}
const TString BROKEN_RESPONSE = "broken";

const TVector<TString> ALL_HOSTS = [] {
    auto result = YASM_MONITORED_HOSTS;
    result.insert(result.end(), YASM_QLOUD_MONITORED_HOSTS.begin(), YASM_QLOUD_MONITORED_HOSTS.end());
    return result;
}();

const TVector<TString> ALL_HOSTS_WITH_C = [] {
    auto result = ALL_HOSTS;
    result.insert(result.end(), CONDUCTOR_HOSTS.begin(), CONDUCTOR_HOSTS.end());
    return result;
}();

class TYasmResolverTest: public ::testing::Test {
public:
    TYasmResolverTest()
        : HttpClient_{CreateCurlClient({}, Registry_)}
    {
    }

    void SetUp() override {
        Server_.Reset(new TTestServer);
        Server_->AddHandler(HOSTS_API, DefaultHostsHandler);
        Server_->AddHandler(CONDUCTOR_API, DefaultConductorHandler);
        UrlLocator_ = StubUrlLocator();
    }

protected:
    TConductorResolverConfig CreateConductorConfig(TString tag) {
        TConductorResolverConfig conf{std::move(tag), HttpClient_.Get()};
        conf.ConductorUrl = Server_->Address();
        return conf;
    }

    TWalleResolverConfig CreateConfig(TVector<TString> tags = {}) {
        TWalleResolverConfig conf{HttpClient_.Get()};
        conf.WalleApiUrl = TStringBuilder() << Server_->Address();
        conf.Tags = std::move(tags);
        return conf;
    }

    IHostGroupResolverPtr CreateResolver(TWalleResolverConfig conf) {
        return CreateYasmAgentGroupResolver(std::move(conf), Registry_, UrlLocator_);
    }

    IHostGroupResolverPtr CreateResolver(TWalleResolverConfig walle, TConductorResolverConfig conductor) {
        return CreateYasmAgentGroupResolver(std::move(walle), std::move(conductor), Registry_, UrlLocator_);
    }

    TVector<TString> OnlyHostnames(TResolveResult r) {
        if (!r.Success()) {
            Cerr << "error: " << r.Error().Message() << Endl;
        }
        Y_ENSURE(r.Success());
        auto hls = r.Extract();
        TVector<TString> result;
        for (auto&& hl: hls) {
            result.emplace_back(hl.Host);
        }

        return result;
    }

protected:
    THolder<TTestServer> Server_;
    IUrlLocatorPtr UrlLocator_;
    TMetricRegistry Registry_;
    IHttpClientPtr HttpClient_;
};

TEST_F(TYasmResolverTest, ResolvesSingleTag) {
    auto r = CreateResolver(CreateConfig({"yasm_monitored"}));
    auto hosts = OnlyHostnames(r->Resolve().ExtractValueSync());

    ASSERT_THAT(hosts, UnorderedElementsAreArray(YASM_MONITORED_HOSTS));
}

TEST_F(TYasmResolverTest, ResolvesMultipleTags) {
    auto r = CreateResolver(CreateConfig({"yasm_monitored", "yasm_qloud_monitored"}));
    auto hosts = OnlyHostnames(r->Resolve().ExtractValueSync());

    ASSERT_THAT(hosts, UnorderedElementsAreArray(ALL_HOSTS));
}

TEST_F(TYasmResolverTest, ReturnsError) {
    {
        auto conf = CreateConfig({"foo"});
        conf.WalleApiUrl = TStringBuilder() << Server_->Address() << "/foo";
        auto r = CreateResolver(conf);

        auto result = r->Resolve().ExtractValueSync();
        ASSERT_THAT(result.Fail(), true);
        ASSERT_THAT(result.Error().IsTransient(), false);
    }
    {
        auto r = CreateResolver(CreateConfig({"yasm_monitored"}));
        Server_->AddHandler(HOSTS_API, [] {
            THttpResponse r;
            r.SetContent(BROKEN_RESPONSE);
            return r;
        });

        auto result = r->Resolve().ExtractValueSync();
        ASSERT_THAT(result.Fail(), true);
        ASSERT_THAT(result.Error().IsTransient(), false);
    }
    {
        auto r = CreateResolver(CreateConfig({"yasm_monitored"}));
        Server_->AddHandler(HOSTS_API, [] {
            THttpResponse r;
            r.SetHttpCode(HTTP_INTERNAL_SERVER_ERROR);
            r.SetContent("internal error message");
            return r;
        });

        auto result = r->Resolve().ExtractValueSync();
        ASSERT_THAT(result.Fail(), true);
        ASSERT_THAT(result.Error().IsTransient(), true);
    }
}

TEST_F(TYasmResolverTest, HostsAreFiltered) {
    struct TLocator: IUrlLocator {
        bool IsLocal(const THostAndLabels& hl) const override {
            return hl.Host == "mrt02e.market.yandex.net";
        }

        void SelectLocal(TVector<THostAndLabels>& hls) const override {
            EraseIf(hls, [this] (auto&& hl) {
                return !IsLocal(hl);
            });
        }
    };

    UrlLocator_ = new TLocator;

    auto r = CreateResolver(CreateConfig({"yasm_monitored", "yasm_qloud_monitored"}));
    auto hosts = OnlyHostnames(r->Resolve().ExtractValueSync());

    ASSERT_THAT(hosts, ElementsAre("mrt02e.market.yandex.net"));
}

TEST_F(TYasmResolverTest, WalleAndConductor) {
    auto r = CreateResolver(
            CreateConfig({"yasm_monitored", "yasm_qloud_monitored"}),
            CreateConductorConfig({"yasm_monitored"})
    );

    auto hosts = OnlyHostnames(r->Resolve().ExtractValueSync());
    ASSERT_THAT(hosts, UnorderedElementsAreArray(ALL_HOSTS_WITH_C));
}

TEST_F(TYasmResolverTest, WalleFiltersByDc) {
    auto conf = CreateConfig({"yasm_monitored"});
    conf.Dc = EDc::IVA;
    auto r = CreateResolver(std::move(conf));
    auto hosts = OnlyHostnames(r->Resolve().ExtractValueSync());

    ASSERT_THAT(hosts, UnorderedElementsAre("mrt02e.market.yandex.net", "iva1-5738.search.yandex.net"));
}
