#include "agents.h"
#include "http_counters.h"
#include "host_resolver_iface.h"

#include <solomon/services/fetcher/lib/dc_label/dc_label.h>

#include <library/cpp/ipv6_address/ipv6_address.h>
#include <library/cpp/http/misc/http_headers.h>
#include <library/cpp/http/misc/httpcodes.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/svnversion/svnversion.h>
#include <library/cpp/threading/future/future.h>
#include <library/cpp/monlib/metrics/metric.h>
#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/resource/resource.h>

#include <util/string/builder.h>
#include <util/string/split.h>
#include <util/string/cast.h>
#include <util/stream/file.h>
#include <util/system/fstat.h>

#include <utility>

namespace NSolomon::NFetcher {
namespace {
    using namespace NThreading;
    using namespace NMonitoring;

    constexpr ui32 DEFAULT_LIMIT{10000};
    const TString HOSTS_PATH{"/v1/hosts"};
    const TString WALLE_API_URL{"https://api.wall-e.yandex-team.ru"};

    struct THostList {
        THostList() = default;
        THostList(TVector<THostAndLabels> hosts)
            : Hosts{std::move(hosts)}
        {
        }

        TVector<THostAndLabels> Hosts;
    };

    const TString VALUE_USER_AGENT = TStringBuilder() << "Solomon-Fetcher/" << GetProgramSvnRevision();

    IHeadersPtr MakeHeaders() {
        return Headers({{TString{NHttpHeaders::USER_AGENT}, VALUE_USER_AGENT}});
    }

    class TWalleRequestContext: public TThrRefBase {
        friend class TWalleClient;
    private:
        void Start() {
            StartRequest();
        }

        void StartRequest() {
            Counters_.StartRequest();
            Client_->Request(MakeRequest(), [self{Self_}] (auto result) {
                self->OnResponse(std::move(result));
            });
        }

        IRequestPtr MakeRequest() {
            TStringBuilder url;
            url << WalleApiUrl_ << HOSTS_PATH
                << "?fields=inv,name,location.short_datacenter_name,ips&strict=true&cursor=" << Cursor_
                << "&tags=" << Tag_
                << "&state__nin=free";

            return Get(url, MakeHeaders());
        }

        void OnResponse(IHttpClient::TResult response) {
            if (response.Fail()) {
                Counters_.CompleteRequestFailure();
                return Complete(TResolveError{
                    TString{"Request failed ("}
                    + ToString(response.Error().Type()) + ')'
                    + ": " + response.Error().Message(), TResolveError::Transient
                });
            }

            const auto code = response.Value()->Code();
            Counters_.CompleteRequest(code);

            if (code == HTTP_OK) {
                // do nothing
            } else if (code != 200) {
                TStringBuilder sb;
                sb << TStringBuf("Request failed with code ") << code;
                auto msg = response.Value()->Data();
                if (!msg.empty()) {
                    sb << ": " << msg;
                }

                return Complete(TResolveError::FromHttpStatus(code, std::move(sb)));
            }

            auto p = ParseResponse(response.Value()->Data());
            if (p.Fail()) {
                return Complete(TResolveError{p.ExtractError()});
            } else if (p.Value()) {
                StartRequest();
            } else {
                Complete(TResolveResult{std::move(Response_.Hosts)});
            }
        }

        void Complete(TResolveResult&& r) {
            Promise_.SetValue(std::move(r));
            Self_.Reset();
        }

        TErrorOr<bool, TGenericError> ParseResponse(TStringBuf body) noexcept {
            using namespace NJson;

            TJsonValue json;
            if (!ReadJsonTree(body, &json)) {
                return TGenericError{"Response is not a valid JSON"};
            }

            TJsonValue::TArray result;
            TString err;
            if (!json.Has(TStringBuf("result"))) {
                err += "Unexpected response format: ";
                err += body.SubStr(0, 200);
            } else if (!json["result"].GetArray(&result)) {
                err += "Expected array, but found ";
                err += ToString(json["result"].GetType());
            }

            if (!err.empty()) {
                return TGenericError{err};
            }

            ui32 recordCount{0};
            TString hostname;
            for (auto&& val: result) {
                TIpv6Address ipAddress;
                if (!(val.Has(TStringBuf("name")) && val["name"].GetString(&hostname))) {
                    err += "Unexpected response format: ";
                    err += body.SubStr(0, 200);
                    return TGenericError{err};
                }

                int inv{0};
                if (!val.Has(TStringBuf("inv")) || (inv = val["inv"].GetIntegerSafe(-1), inv == -1)) {
                    err += "Unexpected response format: ";
                    err += body.SubStr(0, 200);
                    return TGenericError{err};
                }

                TJsonValue::TArray ips;
                if (val["ips"].GetArray(&ips) && ips && ips.size() > 0) {
                    bool ok;
                    ipAddress = TIpv6Address::FromString(ips[0].GetString(), ok);
                }

                if (FilterByLocation(val)) {
                    ++recordCount;
                    auto& h = Response_.Hosts.emplace_back(hostname);
                    if (ipAddress.IsValid()) {
                        h.IpAddress = ipAddress;
                    }
                }

                Cursor_ = Max(Cursor_, inv) + 1;
            }

            RecordsCount_ += recordCount;
            if (RecordsCount_ >= Limit_) {
                return false;
            }

            return recordCount;
        }

        TWalleRequestContext(IHttpClient* client, TStringBuf tag, THttpCounters& counters, ui32 limit, EDc dc)
            : Tag_{tag}
            , Limit_{limit}
            , Counters_{counters}
            , Client_{client}
            , Dc_{dc}
            , DcStr_{to_lower(ToString(Dc_))}
        {
            Self_ = this;
        }

        ~TWalleRequestContext() {
            Y_VERIFY_DEBUG(Promise_.HasValue());
            if (!Promise_.HasValue()) {
                Promise_.SetValue(TResolveResult::FromError("Internal error"));
            }
        }

    private:
        bool FilterByLocation(const NJson::TJsonValue& val) {
            if (Dc_ == EDc::UNKNOWN) {
                return true;
            }

            NJson::TJsonValue dcName;
            if (auto* dcName = val.GetValueByPath(TStringBuf("location.short_datacenter_name"))) {
                return DcStr_ == dcName->GetString();
            }

            return false;
        }

    private:
        TStringBuf Tag_;
        ui32 Limit_{DEFAULT_LIMIT};
        THttpCounters& Counters_;
        TPromise<TResolveResult> Promise_{NewPromise<TResolveResult>()};
        int Cursor_{0};
        IHttpClient* Client_{nullptr};
        EDc Dc_;
        TString DcStr_;
        THostList Response_;
        TIntrusivePtr<TWalleRequestContext> Self_;
        TString WalleApiUrl_;
        size_t RecordsCount_{0};
    };

    class TWalleClient {
    public:
        TWalleClient(IHttpClient* client, THttpCounters& counters, TString walleUrl, EDc dc)
            : HttpClient_{client}
            , Counters_{counters}
            , ApiUrl_{std::move(walleUrl)}
            , Dc_{dc}
        {
        }

        TAsyncResolveResult ResolveTag(const TString& tag) {
            TIntrusivePtr ctx = new TWalleRequestContext(HttpClient_, tag, Counters_, Limit_, Dc_);
            ctx->WalleApiUrl_ = ApiUrl_;
            ctx->Start();
            return ctx->Promise_;
        }

    private:
        IHttpClient* HttpClient_{nullptr};
        THttpCounters& Counters_;
        ui32 Limit_{DEFAULT_LIMIT};
        TString ApiUrl_;
        EDc Dc_;
    };

    struct TWaitContext: TThrRefBase {
        TWaitContext() = default;

        ~TWaitContext() {
            if (Promise_.HasValue()) {
                return;
            }

            Promise_.SetValue(TResolveResult::FromValue(std::move(Urls_)));
        }

        TAsyncResolveResult GetFuture() {
            return Promise_;
        }

        TAsyncResolveResult GetFuture(TResolveResult result) {
            if (Promise_.HasValue()) {
                return Promise_;
            }

            Promise_.SetValue(std::move(result));
            return Promise_;
        }

        void ReportResult(TResolveResult result) noexcept {
            if (Promise_.HasValue()) {
                return;
            }

            if (result.Fail()) {
                Promise_.SetValue(std::move(result));
                return;
            }

            auto hosts = result.Extract();
            Urls_.reserve(Urls_.size() + hosts.size());
            Copy(
                std::make_move_iterator(hosts.begin()),
                std::make_move_iterator(hosts.end()),
                std::back_inserter(Urls_)
            );
        }

    private:
        TPromise<TResolveResult> Promise_{NewPromise<TResolveResult>()};
        TUrls Urls_;
    };

    class TYasmAgentResolver: public IHostGroupResolver {
    public:
        explicit TYasmAgentResolver(TWalleResolverConfig conf, std::optional<TConductorResolverConfig> conductorConfig, TMetricRegistry& registry, IUrlLocatorPtr locator)
            : Config_{std::move(conf)}
            , Locator_{std::move(locator)}
            , WalleCounters_{TStringBuf("walle"), registry}
            , WalleClient_{Config_.Client, WalleCounters_, Config_.WalleApiUrl, Config_.Dc}
        {
            if (conductorConfig) {
                ConductorClient_ = CreateConductorTagResolver(std::move(*conductorConfig), registry);
            }
        }

        TAsyncResolveResult Resolve() noexcept override {
            auto ctx = MakeIntrusive<TWaitContext>();

            for (auto&& tag: Config_.Tags) {
                WalleClient_.ResolveTag(tag).Subscribe([ctx] (auto f) {
                    ctx->ReportResult(f.ExtractValue());
                });
            }

            if (ConductorClient_) {
                ConductorClient_->Resolve().Subscribe([ctx] (auto f) {
                    ctx->ReportResult(f.ExtractValue());
                });
            }

            return ctx->GetFuture().Apply([this] (auto f) -> TResolveResult {
                auto v = f.ExtractValue();
                if (v.Fail()) {
                    return v;
                }

                auto val = v.Extract();
                Locator_->SelectLocal(val);

                return TResolveResult::FromValue(std::move(val));
            });
        }

        const TString& Name() const override {
            return Name_;
        }

    private:
        const TString Name_{"yasm_agent"};
        const TWalleResolverConfig Config_;
        IUrlLocatorPtr Locator_;
        THttpCounters WalleCounters_;
        TWalleClient WalleClient_;
        IHostGroupResolverPtr ConductorClient_;
    };

    class TYasmAgentHostsFileResolverCounters {
    public:
        TYasmAgentHostsFileResolverCounters(NMonitoring::IMetricRegistry* registry) {
            if (registry) {
                AgeSeconds_ = registry->IntGauge(MakeLabels({{"sensor", "yasmHostsFile.ageSeconds"}}));
                ReadErrors_ = registry->IntGauge(MakeLabels({{"sensor", "yasmHostsFile.readErrors"}}));
            }
        }

        void SetAgeSeconds(ui64 duration) {
            if (AgeSeconds_) {
                AgeSeconds_->Set(duration);
            }
        }

        void SetReadErrors(ui64 count) {
            if (ReadErrors_) {
                ReadErrors_->Set(count);
            }
        }

    private:
        NMonitoring::IIntGauge* AgeSeconds_{nullptr};
        NMonitoring::IIntGauge* ReadErrors_{nullptr};
    };

    class TYasmAgentHostsFileResolver: public IHostGroupResolver {
    public:
        TYasmAgentHostsFileResolver(TString filePath, IUrlLocatorPtr locator, TMetricRegistry* registry = nullptr)
            : FilePath_(std::move(filePath))
            , Locator_(std::move(locator))
            , Counters_(registry)
        {}

        const TString& Name() const override {
            return Name_;
        }

        TAsyncResolveResult Resolve() noexcept override {
            auto ctx = MakeIntrusive<TWaitContext>();
            return ctx->GetFuture(DiscoverUrls()).Apply([this] (auto f) -> TResolveResult {
                auto v = f.ExtractValue();
                if (v.Fail()) {
                    return v;
                }

                auto val = v.Extract();
                EraseIf(val, [this] (auto&& hl) {
                    return !Locator_->IsLocal(hl);
                });

                return TResolveResult::FromValue(std::move(val));
            });
        }

    private:
        const TString FilePath_;
        IUrlLocatorPtr Locator_;
        TYasmAgentHostsFileResolverCounters Counters_;
        const TString Name_{"yasm_agent"};
        THashMap<TString, TString> Group2Dc_;

        static constexpr TStringBuf DC_INFO_FILE = "group2dc.txt";

        void LoadDcTable() {
            if (!Group2Dc_.empty()) {
                return;
            }

            TString dcInfo = NResource::Find(DC_INFO_FILE);
            TStringStream s(dcInfo);

            TString line;
            while (s.ReadLine(line)) {
                if (!line) {
                    continue;
                }

                TStringBuf group;
                TStringBuf dc;
                TStringBuf(line).Split(' ', group, dc);

                Group2Dc_[group] = dc;
            }
        }

        static TString CreateGroupLabel(TStringBuf group) {
            return "group_"sv + TString(group);
        }

        static TString CreateMetagroupLabel(TStringBuf metagroup) {
            return "metagroup_"sv + TString(metagroup);
        }

        TVector<THostAndLabels> DiscoverUrls() noexcept {
            LoadDcTable();
            TVector<THostAndLabels> urls;

            bool sorted = true;

            try {
                TFileStat fileStat(FilePath_);
                time_t lastUpdate = Max(fileStat.MTime, fileStat.CTime);
                TDuration fileAge = TInstant::Now() - TInstant::Seconds(lastUpdate);
                Counters_.SetAgeSeconds(fileAge.Seconds());

                TFileInput urls_file(FilePath_);

                TString hostAndLabelsStr;
                NMonitoring::TLabels labels;
                TString lastHost;

                while (urls_file.ReadLine(hostAndLabelsStr)) {
                    if (!hostAndLabelsStr) {
                        continue;
                    }

                    TStringBuf splitter(hostAndLabelsStr.data(), hostAndLabelsStr.size());
                    TStringBuf tmp;
                    TStringBuf host;
                    TStringBuf groupValue;
                    TStringBuf metagroupValue;

                    splitter.Split(' ', host, tmp);
                    tmp.RSplit(' ', groupValue, metagroupValue);

                    if (!host || !groupValue || !metagroupValue) {
                        continue;
                    }

                    if (lastHost > host) {
                        sorted = false;
                    } else if (lastHost && lastHost < host) {
                        urls.emplace_back(TString(lastHost), std::move(labels));
                        labels = {};
                    }

                    auto it = Group2Dc_.find(groupValue);
                    if (it != Group2Dc_.end()) {
                        labels.Add(DC_LABEL_NAME, it->second);
                    }

                    labels.Add(CreateGroupLabel(groupValue), groupValue);
                    labels.Add(CreateMetagroupLabel(metagroupValue), metagroupValue);

                    lastHost = host;
                }

                if (lastHost && !labels.empty()) {
                    urls.emplace_back(lastHost, std::move(labels));
                }

                if (!sorted) {
                    Counters_.SetReadErrors(1u);
                } else {
                    Counters_.SetReadErrors(0u);
                }


            } catch (const yexception& e) {
                Counters_.SetReadErrors(1u);
                Cerr << "Exception while parsing yasm agent hosts file:" << e.what() << '\n';
                return {};
            } catch (...) {
                Counters_.SetReadErrors(1u);
                Cerr << "Unknown exception while parsing yasm agent hosts file\n";
                return {};
            }

            return urls;
        }
    };
} // namespace

IHostGroupResolverPtr CreateYasmAgentGroupResolver(
    TWalleResolverConfig config,
    TConductorResolverConfig conductorConfig,
    NMonitoring::TMetricRegistry& registry,
    IUrlLocatorPtr locator)
{
    return new TYasmAgentResolver{std::move(config), std::move(conductorConfig), registry, std::move(locator)};
}

IHostGroupResolverPtr CreateYasmAgentGroupResolver(
    TWalleResolverConfig config,
    NMonitoring::TMetricRegistry& registry,
    IUrlLocatorPtr locator)
{
    return new TYasmAgentResolver{std::move(config), std::nullopt, registry, std::move(locator)};
}

IHostGroupResolverPtr CreateYasmAgentHostsFileResolver(TString filePath, IUrlLocatorPtr locator, TMetricRegistry* registry) {
    return MakeIntrusive<TYasmAgentHostsFileResolver>(TYasmAgentHostsFileResolver(std::move(filePath), std::move(locator), registry));
}

IHostGroupResolverPtr CreateYasmAgentHostsFileResolver(IUrlLocatorPtr locator, TMetricRegistry* registry) {
    return CreateYasmAgentHostsFileResolver("/Berkanavt/solomon/yasm/hosts.csv", std::move(locator), registry);
}

} // namespace NSolomon::NFetcher
