#include "yd.h"
#include "http_base.h"

#include <infra/yp_service_discovery/api/api.pb.h>

#include <util/generic/guid.h>

namespace NSolomon::NFetcher {
namespace {

using namespace NThreading;
using namespace NMonitoring;

const TString URL_POD = "http://sd.yandex.net:8080/resolve_pods";
const TString URL_ENDPOINT = "http://sd.yandex.net:8080/resolve_endpoints";


TString MakeName(const TYpResolverConfig& conf) {
    if (std::holds_alternative<TYpPodSet>(conf.ClusterConfig.Type)) {
        return TStringBuilder() << "yd-pod:" << conf.ClusterConfig.Cluster << '/'<< std::get<TYpPodSet>(conf.ClusterConfig.Type).Id;
    } else {
        return TStringBuilder() << "yd-endpoint:" << conf.ClusterConfig.Cluster << '/'<< std::get<TYpEndpointSet>(conf.ClusterConfig.Type).Id;
    }
}

class TYdResolver: public IHostGroupResolver, THttpResolverBase {
public:
    TYdResolver(TYpResolverConfig config, TMetricRegistry& registry)
        : THttpResolverBase{config}
        , Config_{std::move(config)}
        , Name_{MakeName(Config_)}
        , Counters_{"yd", registry}
    {
    }

    const TString& Name() const override {
        return Name_;
    }

    TAsyncResolveResult Resolve() noexcept override {
        if (auto* podSet = std::get_if<TYpPodSet>(&Config_.ClusterConfig.Type)) {
            TString reqId = CreateGuidAsString();
            NYP::NServiceDiscovery::NApi::TReqResolvePods payload;
            payload.set_ruid(reqId);
            payload.set_client_name("solomon-fetcher");
            payload.set_pod_set_id(podSet->Id);
            payload.set_cluster_name(Config_.ClusterConfig.Cluster);

            auto req = Post(URL_POD, payload.SerializeAsStringOrThrow(), Headers());

            return DoRequest(std::move(req), [this, reqId] (const TString& result) {
                return ParseYdResponse(result, Config_.ClusterConfig.Labels, reqId);
            }, Counters_);
        }

        auto* endpointSet = std::get_if<TYpEndpointSet>(&Config_.ClusterConfig.Type);

        if (endpointSet == nullptr) {
            return MakeFuture<TResolveResult>(TResolveError{"Pod Set and Endpoint Set are empty"});
        }
        
        TString reqId = CreateGuidAsString();

        NYP::NServiceDiscovery::NApi::TReqResolveEndpoints request;
        request.set_cluster_name(Config_.ClusterConfig.Cluster);
        request.set_endpoint_set_id(endpointSet->Id);
        request.set_client_name("solomon-fetcher");
        request.set_ruid(reqId);

        auto req = Post(URL_ENDPOINT, request.SerializeAsStringOrThrow(), Headers());

        return DoRequest(std::move(req), [this, reqId] (const TString& result) {
            return ParseYpEndpointsResponse(result, Config_.ClusterConfig.Labels, reqId);
        }, Counters_);
    }

private:
    const TYpResolverConfig Config_;
    const TString Name_;
    TCounters Counters_;
};

} // namespace

IHostGroupResolverPtr CreateYdResolver(TYpResolverConfig config, NMonitoring::TMetricRegistry& registry) {
    return MakeIntrusive<TYdResolver>(std::move(config), registry);
}

TResolveResult ParseYdResponse(TStringBuf raw, const NMonitoring::TLabels& additionalLabels, const TString& reqId) noexcept {
    using namespace NYP::NServiceDiscovery;
    try {
        NApi::TRspResolvePods resp;
        if (!resp.ParseFromArray(raw.data(), raw.size())) {
            return TResolveError{TStringBuilder{} << "cannot parse resolve response, reqId=" << reqId};
        }

        if (resp.resolve_status() != NApi::EResolveStatus::OK) {
            return TResolveError{TStringBuilder{}
                    << "non OK resolve status " << NApi::EResolveStatus_Name(resp.resolve_status())
                    << ", reqId=" << reqId};
        }

        TUrls parsedHosts;
        for (const NApi::TPod& pod: resp.pod_set().pods()) {
            for (const auto& location: pod.ip6_address_allocations()) {
                if (location.vlan_id() != TStringBuf{"backbone"} ||
                    location.transient_fqdn().empty() ||
                    location.persistent_fqdn().empty())
                {
                    continue;
                }

                THostAndLabels host{location.persistent_fqdn(), additionalLabels};
                parsedHosts.push_back(std::move(host));
            }
        }
        return TResolveResult::FromValue(parsedHosts);
    } catch (...) {
        TStringBuilder msg;
        msg << "Failed to parse response: " << CurrentExceptionMessage() << ", reqId: " << reqId;
        return TResolveError{msg};
    }
}

TResolveResult ParseYpEndpointsResponse(TStringBuf raw, const NMonitoring::TLabels& additionalLabels, const TString& reqId) noexcept {
    using namespace NYP::NServiceDiscovery;
    try {
        NApi::TRspResolveEndpoints resp;
        if (!resp.ParseFromArray(raw.data(), raw.size())) {
            return TResolveError{TStringBuilder{} << "cannot parse resolve response, reqId=" << reqId};
        }

        if (resp.resolve_status() != NApi::EResolveStatus::OK) {
            return TResolveError{TStringBuilder{}
                    << "non OK resolve status " << NApi::EResolveStatus_Name(resp.resolve_status())
                    << ", reqId=" << reqId};
        }

        TUrls parsedHosts;
        for (const NApi::TEndpoint& endpoint: resp.endpoint_set().endpoints()) {
            ui16 port = static_cast<ui16>(endpoint.Getport());
            THostAndLabels host{endpoint.Getfqdn(), port, {}, additionalLabels};
            parsedHosts.push_back(std::move(host));
        }
        return TResolveResult::FromValue(parsedHosts);
    } catch (...) {
        TStringBuilder msg;
        msg << "Failed to parse response: " << CurrentExceptionMessage() << ", reqId: " << reqId;
        return TResolveError{msg};
    }
}

} // namespace NSolomon::NFetcher
