#include "yp.h"
#include "http_base.h"
#include "host_resolver.h"

#include <library/cpp/json/json_reader.h>
#include <library/cpp/resource/resource.h>

#include <util/generic/hash.h>
#include <util/string/subst.h>
#include <util/string/split.h>

namespace NSolomon::NFetcher {
namespace {
    using namespace NThreading;
    using namespace NMonitoring;

    constexpr auto FILTER_REQUEST_TEMPLATE = R"(
    {
        "object_type": "pod",
        "filter": {
            "query": "_FILTER_"
        },
        "selector": {
            "paths": [
                "/status/ip6_address_allocations"
                _TVM_LABEL_
            ]
        },
        "format": "PF_YSON"
    })";

    constexpr auto HOSTS_INFO_IDX = 0;
    constexpr auto TVM_LABEL_IDX = 1;

    THashMap<TString, TString> MakeUrls() {
        static const TVector<TString> YP_CLUSTERS = [] {
            auto resource = NResource::Find("yp_clusters.txt");
            Y_ENSURE(!resource.empty());
            return StringSplitter(resource).Split('\n').SkipEmpty().ToList<TString>();
        }();

        THashMap<TString, TString> urls;
        for (auto&& cluster: YP_CLUSTERS) {
            urls.emplace(cluster, TStringBuilder() << "https://" << cluster
                << ".yp.yandex.net:8443/ObjectService/SelectObjects");
        }

        return urls;
    }

    TString FilterRequestFromTemplate(TStringBuf filter, TStringBuf tvmLabel) {
        TString request{FILTER_REQUEST_TEMPLATE};
        SubstGlobal(request, "_FILTER_", filter);
        if (!tvmLabel.Empty()) {
            TString tvmLabelStr = TStringBuilder() << ",\"/labels" << tvmLabel << "\"";
            SubstGlobal(request, TStringBuf("_TVM_LABEL_"), tvmLabelStr);
        } else {
            SubstGlobal(request, TStringBuf("_TVM_LABEL_"), tvmLabel);
        }

        return request;
    }

    const TString VLAN = "backbone";

    TString MakeName(const TYpResolverConfig& conf) {
        return TStringBuilder() << "yp-label:" << conf.ClusterConfig.Cluster << '/' << std::get<TYpLabel>(conf.ClusterConfig.Type).Value;
    }

    class TYpResolver: public IHostGroupResolver, THttpResolverBase {
    public:
        TYpResolver(TYpResolverConfig config, TMetricRegistry& registry)
            : THttpResolverBase{config}
            , Config_{std::move(config)}
            , OAuthToken_{Config_.Token}
            , Name_{MakeName(Config_)}
            , Counters_{"yp", registry}
        {
        }

        const TString& Name() const override {
            return Name_;
        }

        TAsyncResolveResult Resolve() noexcept override {
            auto it = ClusterUrls_.find(Config_.ClusterConfig.Cluster);

            if (it == ClusterUrls_.end()) {
                auto p = NewPromise<TUrls>();
                return MakeFuture<TResolveResult>(
                    TResolveError(TStringBuilder() << "unknown YP cluster: " << Config_.ClusterConfig.Cluster)
                );
            }

            return DoRequest(BuildRequest(it->second), [this] (const TString& result) {
                return ParseYpResponse(result, Config_.ClusterConfig.Labels);
            }, Counters_);
        }

    private:
        IRequestPtr BuildRequest(TString url) const {
            TVector<std::pair<TString, TString>> headers = {
                    {"Content-Type", "application/json"},
                    {"Accept", "application/json"},
                    {"Authorization", TString{"OAuth "} + OAuthToken_}
                };

            auto* ypLabel = std::get_if<TYpLabel>(&Config_.ClusterConfig.Type);
            Y_ENSURE(ypLabel && !(ypLabel->Value).empty(), "yp label are not specified for " << Name());

            TString filter = TStringBuilder() << "[" << ypLabel->Value << "]=%true";
            return Post(std::move(url), FilterRequestFromTemplate(filter, ypLabel->TvmLabel), Headers(headers));
        }

    private:
        const TYpResolverConfig Config_;
        static const THashMap<TString, TString> ClusterUrls_;
        const TString OAuthToken_;
        const TString Name_;
        TCounters Counters_;
    };

    const THashMap<TString, TString> TYpResolver::ClusterUrls_ = MakeUrls();

} // namespace

    TResolveResult ParseYpResponse(TStringBuf raw, const TLabels& additionalLabels) noexcept try {
        TUrls parsedHosts;
        NJson::TJsonValue obj;
        NJson::ReadJsonTree(raw, &obj, true);

        for (auto&& result: obj["results"].GetArray()) {
            ui32 tvmDestId{0};
            auto&& payloads = result["value_payloads"].GetArray();

            if (payloads.size() > 1) {
                TryFromString<ui32>(payloads[TVM_LABEL_IDX]["yson"].GetString(), tvmDestId);
            }

            auto&& payload = payloads[HOSTS_INFO_IDX];
            auto elem = payload["yson"];

            for (auto&& val: elem.GetArray()) {
                const auto hasTransientFqdn = val.Has(TStringBuf("transient_fqdn"));
                const auto persistentFqdn = val[(TStringBuf("persistent_fqdn"))].GetString();
                if (val["vlan_id"].GetString() != VLAN || !hasTransientFqdn || persistentFqdn.empty()) {
                    continue;
                }

                THostAndLabels host{persistentFqdn, additionalLabels};
                if (tvmDestId != 0) {
                    host.TvmDestId = tvmDestId;
                }
                parsedHosts.push_back(std::move(host));
            }
        }

        return TResolveResult::FromValue(parsedHosts);
    } catch (...) {
        return TResolveError{TStringBuilder() << "Failed to parse response: " << CurrentExceptionMessage()};
    }

    IHostGroupResolverPtr CreateYpResolver(const TYpResolverConfig& config, TMetricRegistry& registry) {
        return MakeIntrusive<TYpResolver>(config, registry);
    }

} // namespace NSolomon::NFetcher
