#include "host_list_cache.h"

#include <solomon/libs/cpp/actors/events/events.h>
#include <solomon/libs/cpp/logging/logging.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/event_local.h>
#include <library/cpp/actors/core/events.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/actors/core/log.h>

#include <util/generic/algorithm.h>

#include <utility>

namespace NSolomon::NFetcher {
    using namespace NActors;
    using namespace NMonitoring;

namespace {
    class THostListCacheActor: public TActorBootstrapped<THostListCacheActor>, private TPrivateEvents {
        enum {
            EvDone = SpaceBegin,
            End,
        };
        static_assert(End < SpaceEnd, "too many event types");

        struct TEvDone: TEventLocal<TEvDone, EvDone> {
            TEvDone(TVector<THostListCacheItem>&& records)
                : Records{std::move(records)}
            {
            }

            TEvDone(TString error)
                : Error{error}
            {
            }

            TVector<THostListCacheItem> Records;
            std::optional<TString> Error;
        };

    public:
        THostListCacheActor(IHostListCachePtr cache, THostListCacheActorConf conf)
            : Cache_{std::move(cache)}
            , CleanupInterval_{conf.CleanupInterval}
            , RetentionPeriod_{conf.RetentionPeriod}
            , DryRun_{conf.DryRun}
        {
        }

        void Bootstrap(const TActorContext&) {
            Become(&TThis::StateWork);
            SelfPing();
        }

        void SelfPing() {
            Schedule(CleanupInterval_, new TEvents::TEvWakeup);
        }

        STFUNC(StateWork) {
            switch (ev->GetTypeRewrite()) {
                CFunc(TEvents::TSystem::Wakeup, OnWakeup);
                hFunc(TEvDone, OnRequestCompleted);
            }
        }

        void OnWakeup(const TActorContext& ctx) {
            const auto now = ctx.Now();
            const auto since = now - RetentionPeriod_;
            auto* as = ctx.ExecutorThread.ActorSystem;
            const auto selfId = SelfId();

            Cache_->FindNotUpdatedSince(since).Subscribe([as, selfId] (auto f) {
                try {
                    auto records = f.ExtractValue();
                    as->Send(selfId, new TEvDone{std::move(records)});
                } catch (...) {
                    as->Send(selfId, new TEvDone{CurrentExceptionMessage()});
                }
            });
        }

        void OnRequestCompleted(const TEvDone::TPtr& evPtr) {
            auto& ev = *evPtr->Get();
            if (ev.Error) {
                MON_WARN(HostListCache, "Request to host list cache failed: " << *ev.Error);
            } else if (DryRun_) {
                Dump(ev.Records);
            } else {
                TVector<TString> keys;
                keys.resize(ev.Records.size());
                Transform(ev.Records.begin(), ev.Records.end(), keys.begin(), [] (auto&& record) {
                    return record.Id;
                });

                auto* as = TActorContext::ActorSystem();
                Cache_->Delete(keys).Subscribe([as] (auto&& f) {
                    if (!f.HasException()) {
                        return;
                    }

                    try {
                        f.GetValue();
                    } catch (...) {
                        MON_WARN_C(*as, HostListCache, "Error while trying to delete cache records: " << CurrentExceptionMessage());
                    }
                });
            }

            SelfPing();
        }

        void Dump(const TVector<THostListCacheItem>& records) {
            TStringBuilder text;
            text << "Would remove " << records.size() << " records from host list cache: ";
            for (auto&& r: records) {
                text << r.Id << "\t";
            }

            MON_INFO(HostListCache, text);
        }

    private:
        IHostListCachePtr Cache_;
        TDuration CleanupInterval_;
        TDuration RetentionPeriod_;
        bool DryRun_{false};
    };
} // namespace

    IActor* CreateHostListCacheActor(IHostListCachePtr cache, THostListCacheActorConf conf) {
        return new THostListCacheActor{std::move(cache), std::move(conf)};
    }

} // namespace NSolomon::NFetcher
