#pragma once

#include <solomon/services/fetcher/lib/host_groups/host_and_labels.h>

#include <solomon/libs/cpp/actors/fwd.h>
#include <solomon/libs/cpp/ydb/config/ydb_config.pb.h>

#include <library/cpp/threading/future/future.h>

#include <util/generic/maybe.h>
#include <util/generic/ptr.h>
#include <util/generic/array_ref.h>

#include <utility>

namespace NSolomon::NFetcher {
    struct THostListCacheItem {
        THostListCacheItem() = default;
        THostListCacheItem(TString id, const TString& value, TInstant timestampSeconds)
            : THostListCacheItem{std::move(id), HostAndLabelsListFromString(value), timestampSeconds}
        {
        }

        THostListCacheItem(TString id, TVector<THostAndLabels> value, TInstant timestampSeconds)
            : Id{std::move(id)}
            , Value{std::move(value)}
            , Timestamp{timestampSeconds}
        {
        }

        TString Id;
        TVector<THostAndLabels> Value;
        TInstant Timestamp;
    };

    inline IOutputStream& operator<<(IOutputStream& os, const THostListCacheItem& item) {
        os << item.Id << " [";
        for (auto&& h: item.Value) {
            os << h << ", ";
        }

        os << "] " << item.Timestamp.Seconds();
        return os;
    }

    inline bool operator==(const THostListCacheItem& lhs, const THostListCacheItem& rhs) {
        return lhs.Id == rhs.Id && lhs.Value == rhs.Value;
    }

    struct IHostListCache: public TThrRefBase {
        virtual NThreading::TFuture<TMaybe<THostListCacheItem>> Find(const TString& key) const = 0;
        virtual NThreading::TFuture<TVector<THostListCacheItem>> Find(const TVector<TString>& keys) const = 0;
        virtual NThreading::TFuture<void> InsertOrUpdate(const THostListCacheItem& record) = 0;
        virtual NThreading::TFuture<void> Delete(const TVector<TString>& keys) = 0;

        virtual NThreading::TFuture<TVector<THostListCacheItem>> FindNotUpdatedSince(TInstant time) const = 0;

        virtual NThreading::TFuture<void> CreateSchema() = 0;
        virtual NThreading::TFuture<void> DropSchema() = 0;
    };

    using IHostListCachePtr = TIntrusivePtr<IHostListCache>;

    struct TYdbCacheConfig {
        NSolomon::NDb::TYdbConfig YdbConfig;

        TString Path;
    };

    IHostListCachePtr CreateYdbCache(const TYdbCacheConfig& config);
    IHostListCachePtr CreateMemCache();

    struct THostListCacheActorConf {
        // How often to run the cleanup process
        TDuration CleanupInterval = TDuration::Hours(6);

        // If a record hasn't been updated for this long, it should be removed from cache
        TDuration RetentionPeriod = TDuration::Hours(6);

        // Don't actually erase records -- just print to log what would have been deleted
        bool DryRun = false;
    };

    NActors::IActor* CreateHostListCacheActor(IHostListCachePtr cache, THostListCacheActorConf conf);
} // namespace NSolomon::NFetcher
