#include "host_list_cache.h"

#include <library/cpp/threading/future/future.h>
#include <util/system/spinlock.h>

#include <util/generic/hash.h>

namespace NSolomon::NFetcher {
namespace {
    using namespace NThreading;

    class TMemHostListCache: public IHostListCache {
    public:
        // nothing to do for in-memory implementation
        TFuture<void> CreateSchema() override { return MakeFuture(); }
        TFuture<void> DropSchema() override { return MakeFuture(); }

        TFuture<TMaybe<THostListCacheItem>> Find(const TString& key) const override {
            const auto g = Guard(Lock_);
            auto it = Cache_.find(key);
            if (it == Cache_.end()) {
                return MakeFuture<TMaybe<THostListCacheItem>>(Nothing());
            }

            return MakeFuture<TMaybe<THostListCacheItem>>(it->second);
        }

        TFuture<TVector<THostListCacheItem>> Find(const TVector<TString>& keys) const override {
            const auto g = Guard(Lock_);

            TVector<THostListCacheItem> result;
            for (auto&& key: keys) {
                auto it = Cache_.find(key);

                if (it == Cache_.end()) {
                    continue;
                }

                result.push_back(it->second);
            }

            return MakeFuture(result);
        }

        TFuture<void> InsertOrUpdate(const THostListCacheItem& item) override {
            const auto g = Guard(Lock_);

            // Avoid the need of default ctor in the value type
            auto [it, isInserted] = Cache_.emplace(item.Id, item);
            if (!isInserted) {
                it->second = item;
            }

            return MakeFuture();
        }

        TFuture<void> Delete(const TVector<TString>& keys) override {
            const auto g = Guard(Lock_);

            for (auto&& k: keys) {
                Cache_.erase(k);
            }

            return MakeFuture();
        }

        TFuture<TVector<THostListCacheItem>> FindNotUpdatedSince(TInstant time) const override {
            const auto g = Guard(Lock_);
            TVector<THostListCacheItem> result;
            for (auto&& [key, value]: Cache_) {
                if (value.Timestamp < time) {
                    result.push_back(value);
                }
            }

            return MakeFuture(result);
        }

    private:
        TAdaptiveLock Lock_;
        THashMap<TStringBuf, THostListCacheItem> Cache_;
    };
} // namespace

IHostListCachePtr CreateMemCache() {
    return MakeIntrusive<TMemHostListCache>();
}

} // namespace NSolomon::NFetcher
