#include <solomon/services/fetcher/lib/host_list_cache/host_list_cache.h>
#include <solomon/services/fetcher/testlib/db.h>

#include <library/cpp/actors/testlib/test_runtime.h>
#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/testing/gtest/gtest.h>

#include <util/generic/algorithm.h>

#include <utility>

using namespace testing;
using namespace NSolomon::NTesting;
using namespace NSolomon::NFetcher;
using namespace NActors;

namespace NSolomon::NFetcher {
    std::ostream& operator<<(std::ostream& os, const THostListCacheItem& item) {
        TStringBuilder s;
        s << item;
        os << s;
        return os;
    }
} // namespace NSolomon::NFetcher

struct TTestActorRuntime: TTestActorRuntimeBase {
    TTestActorRuntime() {
        TimeProvider = CreateDefaultTimeProvider();
    }
};

void WaitForBootstrap(TTestActorRuntimeBase& runtime) {
    TDispatchOptions options;
    options.FinalEvents.emplace_back(TEvents::TSystem::Bootstrap, 1);

    ASSERT_TRUE(runtime.DispatchEvents(options));
}

struct TListeningCacheProxy: IHostListCache{
    TListeningCacheProxy(IHostListCachePtr impl)
        : Impl{std::move(impl)}
    {
    }

    NThreading::TFuture<void> WaitForDelete() {
        return DeleteCalled.GetFuture();
    }

    NThreading::TFuture<TMaybe<THostListCacheItem>> Find(const TString& key) const override {
        return Impl->Find(key);
    }

    NThreading::TFuture<TVector<THostListCacheItem>> Find(const TVector<TString>& keys) const override {
        return Impl->Find(keys);
    }

    NThreading::TFuture<void> InsertOrUpdate(const THostListCacheItem& record) override {
        return Impl->InsertOrUpdate(record);
    }

    NThreading::TFuture<void> Delete(const TVector<TString>& keys) override {
        auto f = Impl->Delete(keys);
        f.Subscribe([this] (auto){
            DeleteCalled.SetValue();
        });

        return f;
    }

    NThreading::TFuture<TVector<THostListCacheItem>> FindNotUpdatedSince(TInstant time) const override {
        return Impl->FindNotUpdatedSince(time);
    }

    NThreading::TFuture<void> CreateSchema() override {
        return Impl->CreateSchema();
    }

    NThreading::TFuture<void> DropSchema() override {
        return Impl->DropSchema();
    }

    NThreading::TPromise<void> DeleteCalled{NThreading::NewPromise<void>()};
    IHostListCachePtr Impl;
};

class THostListCacheTest: public ::testing::Test {
public:
    void SetUp() override {
        TString guid = TStringBuilder() << CreateGuidAsString();
        TString prefix = TStringBuilder() << '/' << Db_.Config().GetDatabase() << '/' << guid;
        Db_.MakeDirectory(prefix);
        Cache_ = CreateYdbCache(TYdbCacheConfig{
            .YdbConfig = Db_.Config(),
            .Path = prefix + "/hostListCache",
        });

        Cache_->CreateSchema().GetValueSync();
    }

    void TearDown() override {
        Cache_->DropSchema().GetValueSync();
    }

protected:
    IHostListCachePtr Cache_;
    TTestDb Db_;
};

TEST_F(THostListCacheTest, FindSinglePresent) {
    auto expected = THostListCacheItem{
            "foo",
            "bar",
            TInstant::Seconds(42),
    };

    Cache_->InsertOrUpdate(expected).GetValueSync();

    auto item = Cache_->Find("foo").GetValueSync();
    ASSERT_TRUE(item.Defined());
    ASSERT_THAT(*item, Eq(expected));
}

TEST_F(THostListCacheTest, FindSingleAbsent) {
    auto item = Cache_->Find("foo").GetValueSync();
    ASSERT_FALSE(item.Defined());
}

TEST_F(THostListCacheTest, FindMany) {
    TVector<THostListCacheItem> items {
            {"a", "value1", TInstant::Seconds(41)},
            {"b", "value2", TInstant::Seconds(42)},
    };

    for (auto&& it: items) {
        Cache_->InsertOrUpdate(it).GetValueSync();
    }

    TVector<TString> lookup{"a", "c"};
    auto result = Cache_->Find(lookup).GetValueSync();

    ASSERT_THAT(result, ElementsAreArray({items[0]}));
}

TEST_F(THostListCacheTest, Update) {
    auto value = THostListCacheItem{
            "foo",
            "bar",
            TInstant::Seconds(42),
    };

    Cache_->InsertOrUpdate(value).GetValueSync();
    value.Value = {THostAndLabels::FromString("baz")};
    Cache_->InsertOrUpdate(value).GetValueSync();

    auto item = Cache_->Find("foo").GetValueSync();
    ASSERT_TRUE(item.Defined());
    ASSERT_THAT(*item, Eq(value));
}

TEST_F(THostListCacheTest, Delete) {
    auto expected = THostListCacheItem{
            "foo",
            "bar",
            TInstant::Seconds(42),
    };

    Cache_->InsertOrUpdate(expected).GetValueSync();

    auto item = Cache_->Find("foo").GetValueSync();
    ASSERT_TRUE(item.Defined());

    Cache_->Delete({item->Id}).GetValueSync();
    item = Cache_->Find("foo").GetValueSync();
    ASSERT_FALSE(item.Defined());
}

TEST_F(THostListCacheTest, FindOutdated) {
    THostListCacheActorConf conf;
    const auto now = TInstant::Now();
    ASSERT_THAT(now, Gt(TInstant::Seconds(42) + conf.RetentionPeriod));

    TVector<THostListCacheItem> values {
            {"outdated", "bar", TInstant::Seconds(42)},
            {"outdated2", "bar", TInstant::Seconds(0)},
            {"new", "bar", now},
            {"new2", "bar", now - TDuration::Seconds(5)},
    };

    const TVector<THostListCacheItem> expected = [vs=values, t=(now - conf.RetentionPeriod)] () mutable {
        EraseIf(vs, [t] (auto v) {
            return v.Timestamp < t;
        });

        return vs;
    }();

    for (auto&& v: values) {
        Cache_->InsertOrUpdate(v).GetValueSync();
    }

    TTestActorRuntimeBase actorSystem;
    actorSystem.Initialize();
    actorSystem.UpdateCurrentTime(now);

    TIntrusivePtr<TListeningCacheProxy> cacheProxy{new TListeningCacheProxy{Cache_}};
    auto* cacheCleaner = CreateHostListCacheActor(cacheProxy, {});
    const auto cleanerId = actorSystem.Register(cacheCleaner);
    WaitForBootstrap(actorSystem);

    actorSystem.Send(new IEventHandle(cleanerId, TActorId(), new TEvents::TEvWakeup));
    actorSystem.DispatchEvents({}, TDuration::Seconds(5));

    TVector<TString> keys(values.size());
    Transform(values.begin(), values.end(), keys.begin(), [] (auto&& v) {
        return v.Id;
    });

    cacheProxy->WaitForDelete().GetValueSync();
    auto records = Cache_->Find(keys).GetValueSync();
    ASSERT_THAT(records, UnorderedElementsAreArray(expected));
};
