#include "host_list_cache.h"

#include <ydb/public/sdk/cpp/client/ydb_driver/driver.h>
#include <ydb/public/sdk/cpp/client/ydb_table/table.h>
#include <ydb/public/sdk/cpp/client/ydb_scheme/scheme.h>
#include <ydb/public/sdk/cpp/client/ydb_result/result.h>

#include <solomon/libs/cpp/ydb/util.h>

#include <util/string/printf.h>

namespace NSolomon::NFetcher {
namespace {

    const auto RETRY_SETTINGS = NYdb::NTable::TRetryOperationSettings()
        .MaxRetries(2)
        .SlowBackoffSettings(
            NYdb::NTable::TRetryOperationSettings::DefaultSlowBackoffSettings()
                .SlotDuration(TDuration::MilliSeconds(500))
        );

    constexpr auto FIND_QUERY = R"(
        --!syntax_v1
        DECLARE $id as Utf8;

        SELECT id, value, createdSeconds
        FROM `%s`
        WHERE id = $id;)";

    constexpr auto UPSERT_QUERY = R"(
        --!syntax_v1
        DECLARE $id as Utf8;
        DECLARE $value as Utf8;
        DECLARE $ts as Uint64;

        UPSERT INTO `%s` (
            id,
            value,
            createdSeconds
        )
        VALUES (
            $id,
            $value,
            $ts
        );)";

    constexpr auto FIND_MANY_QUERY = R"(
        --!syntax_v1
        DECLARE $ids as List<Utf8>;

        SELECT id, value, createdSeconds
        FROM `%s`
        WHERE id IN $ids;)";

    constexpr auto DELETE_QUERY = R"(
        --!syntax_v1
        DECLARE $ids as List<Utf8>;

        DELETE FROM `%s`
        WHERE id IN $ids;)";

    constexpr auto FIND_OUTDATED_QUERY = R"(
        --!syntax_v1
        DECLARE $ts as Uint64;

        SELECT id, value, createdSeconds
        FROM `%s`
        WHERE createdSeconds < $ts;)";

    using namespace NThreading;
    using namespace NYdb;
    using namespace NYdb::NTable;

    void MakeParamsFromIdList(const TVector<TString>& keys, TParamsBuilder& pbuilder) {
        auto& ids = pbuilder.AddParam("$ids").BeginList();

        for (auto&& k: keys) {
            ids.AddListItem().Utf8(k);
        }

        ids.EndList().Build();
    }

    TVector<THostListCacheItem> ParseSelectResult(TResultSets rs) {
        Y_ENSURE(rs.size() == 1, "Expected single result set");
        TResultSetParser rsp{rs[0]};

        TVector<THostListCacheItem> items;
        items.reserve(rs[0].RowsCount());
        while (rsp.TryNextRow()) {
            items.emplace_back(
                *rsp.ColumnParser(0).GetOptionalUtf8(),
                *rsp.ColumnParser(1).GetOptionalUtf8(),
                TInstant::Seconds(*rsp.ColumnParser(2).GetOptionalUint64())
            );
        }

        return items;
    }

    class TYdbCache: public IHostListCache {
    public:
        explicit TYdbCache(const TYdbCacheConfig& conf)
            : Conf_{conf.YdbConfig}
            , Driver_{NYdb::TDriverConfig{}
                .SetEndpoint(Conf_.GetAddress())
                .SetDatabase(Conf_.GetDatabase())}
            , Client_{Driver_}
            , Path_{conf.Path}
            , Queries_{
                .FindOne = Sprintf(FIND_QUERY, Path_.c_str()),
                .FindMany = Sprintf(FIND_MANY_QUERY, Path_.c_str()),
                .FindOutdated = Sprintf(FIND_OUTDATED_QUERY, Path_.c_str()),
                .Insert = Sprintf(UPSERT_QUERY, Path_.c_str()),
                .Delete = Sprintf(DELETE_QUERY, Path_.c_str()),
            }
        {
        }

        TFuture<void> CreateSchema() override {
            auto future = NDb::Execute<NYdb::TAsyncStatus>(Client_, [=] (TSession session) {
                auto td = TTableBuilder{}
                    .AddNullableColumn("id", NYdb::EPrimitiveType::Utf8)
                    .AddNullableColumn("value", NYdb::EPrimitiveType::Utf8)
                    .AddNullableColumn("createdSeconds", NYdb::EPrimitiveType::Uint64)
                    .SetPrimaryKeyColumn("id")
                    .Build();

                return session.CreateTable(Path_, std::move(td));
            });
            return NDb::CheckStatus("cannot create schema", future);
        };

        TFuture<void> DropSchema() override {
            auto future = NDb::Execute<NYdb::TAsyncStatus>(Client_, [=] (TSession session) {
                return session.DropTable(Path_);
            });
            return NDb::CheckStatus("cannot drop schema", future);
        }

        TFuture<TMaybe<THostListCacheItem>> Find(const TString& key) const override {
            auto params = TParamsBuilder{}
                .AddParam("$id").Utf8(key).Build()
                .Build();

            return NDb::ExecutePreparedWithRetries(Client_, Queries_.FindOne, std::move(params), RETRY_SETTINGS)
                .Apply([] (const TAsyncDataQueryResult& f) {
                    auto&& v = f.GetValueSync();

                    if (!v.IsSuccess()) {
                        ythrow yexception() << v.GetIssues().ToString();
                    }

                    auto&& rs = v.GetResultSets();
                    if (rs.empty() || rs[0].RowsCount() == 0) {
                        return MakeFuture<TMaybe<THostListCacheItem>>(Nothing());
                    }

                    Y_ENSURE(rs.size() == 1 && rs[0].RowsCount() == 1, "Expected single value");
                    TResultSetParser rsp{rs[0]};

                    rsp.TryNextRow();
                    THostListCacheItem item{
                        *rsp.ColumnParser(0).GetOptionalUtf8(),
                        *rsp.ColumnParser(1).GetOptionalUtf8(),
                        TInstant::Seconds(*rsp.ColumnParser(2).GetOptionalUint64())
                    };

                    return MakeFuture<TMaybe<THostListCacheItem>>(item);
                });
        }

        TFuture<TVector<THostListCacheItem>> Find(const TVector<TString>& keys) const override {
            TParamsBuilder pbuilder;
            MakeParamsFromIdList(keys, pbuilder);
            auto params = pbuilder.Build();

            if (keys.empty()) {
                return MakeFuture<TVector<THostListCacheItem>>();
            }

            return NDb::ExecutePreparedWithRetries(Client_, Queries_.FindMany, std::move(params), RETRY_SETTINGS)
                .Apply([] (const TAsyncDataQueryResult& f) {
                    auto&& v = f.GetValueSync();

                    if (!v.IsSuccess()) {
                        ythrow yexception() << v.GetIssues().ToString();
                    }

                    auto&& rs = v.GetResultSets();
                    if (rs.empty() || rs[0].RowsCount() == 0) {
                        return MakeFuture<TVector<THostListCacheItem>>();
                    }

                    auto items = ParseSelectResult(rs);
                    return MakeFuture(items);
                });
        }

        TFuture<void> InsertOrUpdate(const THostListCacheItem& item) override {
            auto params = TParamsBuilder{}
                .AddParam("$id").Utf8(item.Id).Build()
                .AddParam("$value").Utf8(HostAndLabelsListToString(item.Value)).Build()
                .AddParam("$ts").Uint64(item.Timestamp.Seconds()).Build()
            .Build();

            auto future = NDb::ExecutePreparedWithRetries(Client_, Queries_.Insert, std::move(params), RETRY_SETTINGS);
            return NDb::CheckStatus("cannot insert or update host list cache item", future);
        }

        TFuture<void> Delete(const TVector<TString>& keys) override {
            if (keys.empty()) {
                return MakeFuture();
            }

            TParamsBuilder pbuilder;
            MakeParamsFromIdList(keys, pbuilder);
            auto params = pbuilder.Build();

            return NDb::ExecutePreparedWithRetries(Client_, Queries_.Delete, std::move(params), RETRY_SETTINGS)
                .Apply([] (const TAsyncDataQueryResult& f) {
                    auto&& v = f.GetValueSync();

                    if (!v.IsSuccess()) {
                        ythrow yexception() << v.GetIssues().ToString();
                    }
                });
        }

        TFuture<TVector<THostListCacheItem>> FindNotUpdatedSince(TInstant time) const override {
            auto params = TParamsBuilder{}
                .AddParam("$ts").Uint64(time.Seconds()).Build()
            .Build();

            return NDb::ExecutePreparedWithRetries(Client_, Queries_.FindOutdated, std::move(params), RETRY_SETTINGS)
                .Apply([] (TAsyncDataQueryResult f) {
                    auto&& v = f.ExtractValue();

                    if (!v.IsSuccess()) {
                        ythrow yexception() << v.GetIssues().ToString();
                    }

                    auto&& rs = v.GetResultSets();
                    if (rs.empty() || rs[0].RowsCount() == 0) {
                        return MakeFuture<TVector<THostListCacheItem>>();
                    }

                    auto items = ParseSelectResult(rs);
                    return MakeFuture<TVector<THostListCacheItem>>(items);
                });
        }

    private:
        const NSolomon::NDb::TYdbConfig Conf_;
        const NYdb::TDriver Driver_;
        NYdb::NTable::TTableClient Client_;
        TString Path_;

        const struct {
            TString FindOne;
            TString FindMany;
            TString FindOutdated;
            TString Insert;
            TString Delete;
        } Queries_;
    };
}

IHostListCachePtr CreateYdbCache(const TYdbCacheConfig& config) {
    return MakeIntrusive<TYdbCache>(config);
}

} // NSolomon::NFetcher
