#pragma once

#include <util/generic/string.h>
#include <util/string/builder.h>
#include <util/generic/hash.h>

#include <optional>

namespace NSolomon::NFetcher {
    class TProjectEntityId {
    public:
        TProjectEntityId() = default;
        TProjectEntityId(TString id, TString projectId)
            : Id_{std::move(id)}
            , ProjectId_{std::move(projectId)}
        {
        }

        TString ToString() const noexcept {
            return TStringBuilder() << ProjectId_ << '/'<< Id_;
        }

        bool IsValid() const noexcept {
            return !ProjectId_.Empty() && !Id_.Empty();
        }

        bool IsEmpty() const noexcept {
            return ProjectId_.Empty() && Id_.Empty();
        }

        size_t Hash() const noexcept {
            return CombineHashes(
                THash<TString>{}(Id_),
                THash<TString>{}(ProjectId_)
            );
        }

        const TString& ProjectId() const noexcept {
            return ProjectId_;
        }

    protected:
        bool Equals(const TProjectEntityId& other) const noexcept {
            return other.Id_ == Id_ && other.ProjectId_ == ProjectId_;
        }

        const TString& EntityId() const {
            return Id_;
        }

    private:
        TString Id_;
        TString ProjectId_;
    };

    // NB: we may have used Id/ProjectId as a key here as well,
    // but using shard's numeric id (which is guaranteed to be unique) is more efficient
    class TShardId {
    public:
        using TNumId = ui32;

        TShardId() = default;

        TShardId(TString id, TNumId numId)
            : Id_{std::move(id)}
            , NumId_{numId}
        {
        }

        TShardId(TNumId numId)
            : NumId_{numId}
        {
        }

        bool IsValid() const noexcept {
            return !Id_.Empty() && NumId_ != 0;
        }

        TString ToString() const {
            return TStringBuilder() << NumId_ << " (" << (Id_.Empty() ? TString{"?"} : Id_) << ')';
        }

        TNumId NumId() const noexcept {
            return NumId_;
        }

        const TString& StrId() const noexcept {
            return Id_;
        }

        size_t Hash() const noexcept {
            return NumId_;
        }

        friend bool operator==(const TShardId& lhs, const TShardId& rhs);

    private:
        TString Id_;
        TNumId NumId_{0};
    };

    struct TClusterId: TProjectEntityId {
        using TProjectEntityId::TProjectEntityId;

        const TString& ClusterId() const {
            return EntityId();
        }

        friend bool operator==(const TClusterId& lhs, const TClusterId& rhs);
    };

    struct TServiceId: TProjectEntityId {
        using TProjectEntityId::TProjectEntityId;

        const TString& ServiceId() const {
            return EntityId();
        }

        friend bool operator==(const TServiceId& lhs, const TServiceId& rhs);
    };

    using TProviderId = TString;

    inline bool operator==(const TClusterId& lhs, const TClusterId& rhs) {
        return lhs.Equals(rhs);
    }

    inline bool operator==(const TServiceId& lhs, const TServiceId& rhs) {
        return lhs.Equals(rhs);
    }

    inline bool operator==(const TShardId& lhs, const TShardId& rhs) {
        if (lhs.NumId()) {
            return lhs.NumId_ == rhs.NumId_;
        }

        return lhs.StrId() == rhs.StrId();
    }

    inline IOutputStream& operator<<(IOutputStream& os, const TClusterId& id) {
        os << id.ToString();
        return os;
    }

    inline IOutputStream& operator<<(IOutputStream& os, const TServiceId& id) {
        os << id.ToString();
        return os;
    }

    inline IOutputStream& operator<<(IOutputStream& os, const TShardId& id) {
        os << id.ToString();
        return os;
    }
} // namespace NSolomon::NFetcher

template <>
struct THash<NSolomon::NFetcher::TClusterId> {
    size_t operator()(const NSolomon::NFetcher::TClusterId& id) const noexcept {
        return id.Hash();
    }
};

template <>
struct THash<NSolomon::NFetcher::TServiceId> {
    size_t operator()(const NSolomon::NFetcher::TServiceId& id) const noexcept {
        return id.Hash();
    }
};

template <>
struct THash<NSolomon::NFetcher::TShardId> {
    size_t operator()(const NSolomon::NFetcher::TShardId& id) const noexcept {
        return id.Hash();
    }
};
