#include "assignments_requester.h"

#include <solomon/libs/cpp/logging/logging.h>

#include <library/cpp/actors/core/event_local.h>
#include <library/cpp/actors/core/hfunc.h>

#include <util/random/random.h>
#include <util/system/types.h>

using namespace NActors;
using namespace NSolomon::NIngestor;
using namespace yandex::monitoring::ingestor;

namespace NSolomon::NFetcher {
namespace {

class TLocalEvents: private TPrivateEvents {
private:
    enum {
        Request = SpaceBegin,
        Response,
        Error,
        NodeUpdate,
        End,
    };

    static_assert(End < SpaceEnd, "too many event types");

public:
    struct TRequest: public TEventLocal<TRequest, Request> {
    };

    struct TError: public TEventLocal<TError, Error> {
        TString Msg;

        explicit TError(TString msg)
            : Msg{std::move(msg)}
        {}
    };

    struct TResponse: public TEventLocal<TResponse, Response> {
        TGetAssignedShardsResponseOrErrorPtr Response;

        explicit TResponse(TGetAssignedShardsResponseOrErrorPtr&& response)
            : Response{std::move(response)}
        {}
    };


    struct TNodeUpdate: public TEventLocal<TNodeUpdate, NodeUpdate> {
        TClusterNode Node;
        TGetAssignedShardsResponseOrErrorPtr Response;

        explicit TNodeUpdate(TClusterNode node, TGetAssignedShardsResponseOrErrorPtr&& response)
            : Node{std::move(node)}
            , Response{std::move(response)}
        {}
    };
};

// IAR -- Ingestor Assignments Requester
#define IAR_WARN(...) MON_WARN(IngestorAssignmentsRequester, __VA_ARGS__)

class TIngestorNodeRequester: public TActorBootstrapped<TIngestorNodeRequester> {
public:
    explicit TIngestorNodeRequester(
            IIngestorClient* client,
            TClusterNode node,
            TActorId parent,
            TDuration updateInterval)
        : Client_{std::move(client)}
        , Node_{std::move(node)}
        , Parent_{std::move(parent)}
        , UpdateInterval_{updateInterval}
        , Delay_{UpdateInterval_}
        , ActorSystem_{TActorContext::ActorSystem()}
    {}


    void Bootstrap() {
        OnRequest();
    }

private:
    STATEFN(Sleeping) {
        switch (ev->GetTypeRewrite()) {
            sFunc(TLocalEvents::TRequest, OnRequest);
            sFunc(TEvents::TEvPoison, OnDestruction)
        }
    }

    STATEFN(WaitingForAResponse) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TLocalEvents::TResponse, OnResponse);
            hFunc(TLocalEvents::TError, OnError);
            sFunc(TEvents::TEvPoison, OnDestruction)
        }
    }

    void OnRequest() {
        Become(&TThis::WaitingForAResponse);

        // TODO(ivanzhukov): timeout
        yandex::monitoring::ingestor::TGetAssignedShardsRequest req;
        Client_->GetAssignedShards(req).Subscribe([as{ActorSystem_}, self{SelfId()}](TAsyncGetAssignedShardsResponse f) {
            try {
                auto resultOrErr = f.ExtractValueSync();

                if (resultOrErr->Fail()) {
                    as->Send(self, new TLocalEvents::TError{ resultOrErr->Error().Msg });
                    return;
                }

                as->Send(self, new TLocalEvents::TResponse{ std::move(resultOrErr) });
            } catch (...) {
                as->Send(self, new TLocalEvents::TError{ CurrentExceptionMessage() });
            }
        });
    }

    void OnError(const TLocalEvents::TError::TPtr& evPtr) {
        IAR_WARN("failed to get assignments from " << Node_.Endpoint << ": " << evPtr->Get()->Msg);

        Become(&TThis::Sleeping);
        auto jitter = TDuration::MilliSeconds(RandomNumber(2'000ul));
        Delay_ = Min(1.5 * Delay_ + jitter, TDuration::Seconds(10));

        Schedule(Delay_, new TLocalEvents::TRequest);
    }

    void OnResponse(const TLocalEvents::TResponse::TPtr& evPtr) {
        Send(Parent_, new TLocalEvents::TNodeUpdate{ Node_, std::move(evPtr->Get()->Response) });

        Delay_ = UpdateInterval_;
        Become(&TThis::Sleeping);
        Schedule(Delay_, new TLocalEvents::TRequest);
    }

    void OnDestruction() {
        // TODO: send an event to the parent
        PassAway();
    }

private:
    IIngestorClient* Client_;
    TClusterNode Node_;
    TActorId Parent_;
    TDuration UpdateInterval_;
    TDuration Delay_;

    TActorSystem* ActorSystem_;
};

class TIngestorClusterRequester: public TActorBootstrapped<TIngestorClusterRequester> {
public:
    TIngestorClusterRequester(
            IClusterMapPtr cluster,
            IIngestorClusterClientPtr clients,
            TDuration updateInterval)
        // TODO(ivanzhukov): auto-update cluster
        : Cluster_{std::move(cluster)}
        , Clients_{std::move(clients)}
        , UpdateInterval_{updateInterval}
    {}

    void Bootstrap() {
        Become(&TThis::State);

        for (const auto& node: Cluster_->Nodes()) {
            auto client = Clients_->Get(node.Endpoint);

            Register(new TIngestorNodeRequester{
                    client,
                    node,
                    SelfId(),
                    UpdateInterval_,
            });
        }
    }

private:
    STATEFN(State) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TIngestorRequesterEvents::TRequestAssignments, OnAssignmentsRequest);
            hFunc(TLocalEvents::TNodeUpdate, OnNodeUpdate);
            sFunc(TEvents::TEvPoison, OnDestruction)
        }
    }

    void OnNodeUpdate(const TLocalEvents::TNodeUpdate::TPtr& evPtr) {
        const auto& ev = *evPtr->Get();
        auto&& val = ev.Response->Extract();

        auto it = NodeToAssignedShards_.find(ev.Node);
        TVector<ui32>& numIds = it == NodeToAssignedShards_.end()
                ? NodeToAssignedShards_.emplace(ev.Node, ::Reserve(val.NumIdsSize())).first->second
                : it->second;

        numIds.clear();

        // XXX(ivanzhukov): what if size == 0?
        for (auto numId: val.GetNumIds()) {
            numIds.emplace_back(numId);
        }
    }

    void OnAssignmentsRequest(const TIngestorRequesterEvents::TRequestAssignments::TPtr& evPtr) {
        TVector<TShardAssignment> result(::Reserve(NodeToAssignedShards_.size()));

        for (const auto& [node, numIds]: NodeToAssignedShards_) {
            result.emplace_back(node, numIds);
        }

        Send(evPtr->Sender, new TIngestorRequesterEvents::TAssignmentsResponse{
            std::move(result),
        });
    }

    void OnDestruction() {
        // TODO: wait for a complete children destruction
        PassAway();
    }

private:
    IClusterMapPtr Cluster_;
    IIngestorClusterClientPtr Clients_;
    TDuration UpdateInterval_;

    THashMap<TClusterNode, TVector<ui32>> NodeToAssignedShards_;
};

#undef IAR_WARN


} // namespace

IActor* CreateIngestorClusterRequester(
        IClusterMapPtr cluster,
        IIngestorClusterClientPtr clients,
        TDuration updateInterval)
{
    return new TIngestorClusterRequester{
            std::move(cluster),
            std::move(clients),
            updateInterval,
    };
}

} // namespace NSolomon::NFetcher
