#include "load_balancer.h"
#include "helper.h"

#include <solomon/services/fetcher/config/load_balancer_config.pb.h>
#include <solomon/services/fetcher/lib/host_groups/host_and_labels.h>

#include <solomon/libs/cpp/config/units.h>
#include <solomon/libs/cpp/coordination/load_balancer/load_balancer.h>
#include <solomon/libs/cpp/distributed_lock/lock.h>
#include <solomon/libs/cpp/sync/rw_lock.h>

#include <ydb/public/sdk/cpp/client/ydb_driver/driver.h>
#include <ydb/public/sdk/cpp/client/ydb_coordination/coordination.h>

#include <library/cpp/actors/core/actorsystem.h>
#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>

#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/generic/algorithm.h>
#include <util/system/hostname.h>

using namespace NActors;
using namespace NMonitoring;
using namespace NSolomon;
using namespace NSolomon::NCoordination;
using namespace NYdb;

namespace NSolomon::NFetcher {
namespace {
    class TUrlLocatorActor: public TActorBootstrapped<TUrlLocatorActor> {
    public:
        TUrlLocatorActor(TActorId balancerId, TActorId resolverId, IFetcherClusterPtr cluster, IAssignmentsConsumerPtr consumer)
            : BalancerId_{balancerId}
            , ResolverId_{resolverId}
            , Cluster_{std::move(cluster)}
            , Consumer_{std::move(consumer)}
        {
        }

        void Bootstrap() {
            TActivationContext::Schedule(
                TDuration::Seconds(5),
                new IEventHandle{BalancerId_, SelfId(), new TEvents::TEvSubscribe}
            );

            Become(&TThis::StateWork);
        }

        STFUNC(StateWork) {
            Y_UNUSED(ctx);
            switch (ev->GetTypeRewrite()) {
                hFunc(TLoadBalancerEvents::TEvAssignment, OnAssignments);
                cFunc(TEvents::TEvPoison::EventType, PassAway);
            }
        }

        void OnAssignments(const TLoadBalancerEvents::TEvAssignment::TPtr& ev) {
            auto assignment = ev->Get()->Assignment;
            bool needResolve = Cluster_ && assignment.Slices() != LastAssignment_;
            if (needResolve) {
                LastAssignment_ = assignment.Slices();
            }

            Consumer_->OnAssignments(
                std::move(assignment)
            );

            if (needResolve) {
                Send(ResolverId_, new TEvResolveCluster(Cluster_, TEvResolveCluster::ESubscriptionType::Forced));
            }
        }

    private:
        TActorId BalancerId_;
        TActorId ResolverId_;
        IFetcherClusterPtr Cluster_;
        IAssignmentsConsumerPtr Consumer_;
        TSlices LastAssignment_;
    };

    class TUrlLocator: public IAssignmentsConsumer {
    public:
        void OnAssignments(TAssignments&& assn) override {
            auto impl = Impl_.Write();
            impl->SetAssignments(std::move(assn));
        }

        bool IsLocal(const THostAndLabels& hl) const override {
            auto impl = Impl_.Read();
            return IsLocalCheck(*impl, hl);
        }

        void SelectLocal(TVector<THostAndLabels>& hls) const override {
            auto impl = Impl_.Read();
            EraseIf(hls, [&] (auto&& hl) {
                return !IsLocalCheck(*impl, hl);
            });
        }

    private:
        static bool IsLocalCheck(const TLocatorHelper& locator, const THostAndLabels& hl) {
            if (locator.IsEmpty()) {
                return false;
            }
            return locator.IsLocal(hl.UrlHash());
        }

    private:
        NSync::TLightRwLock<TLocatorHelper> Impl_;
    };
} // namespace

    IDistributedLockPtr CreateLock(const NYdb::TDriver& driver, const TLockConfig& lockConf) {
        auto path = lockConf.GetNodePath();
        Y_ENSURE(!path.Empty());
        auto name = lockConf.GetSemaphoreName();
        Y_ENSURE(!name.Empty());
        auto data = FQDNHostName();

        TYdbLockConfig config{
            .Driver = driver,
            .Path = path,
            .Name = name,
            .Data = data,
        };

        auto f = PrepareYdbLock(config);
        auto ok = f.Wait(
            TDuration::Minutes(1)
        );

        Y_ENSURE(ok, "YDB prepare timed out");
        return CreateYdbLock(config);
    }

    NCoordination::TLoadBalancerConfig CreateLoadBalancerConfig(
        IMetricFactory& metricFactory,
        const TLoadBalancerConfig& protoConf)
    {
        NCoordination::TLoadBalancerConfig result{
            .MetricFactory = metricFactory,
        };

        if (protoConf.HasPingInterval()) {
            result.PingInterval = FromProtoTime(protoConf.GetPingInterval());
        }

        if (protoConf.HasOfflineThreshold()) {
            result.OfflineThreshold = FromProtoTime(protoConf.GetOfflineThreshold());
        }

        return result;
    }

    IAssignmentsConsumerPtr CreateAssignmentsConsumer() {
        return new TUrlLocator;
    }

    IActor* CreateLoadBalancerActor(
            TActorId balancerId,
            TActorId resolverId,
            IFetcherClusterPtr cluster,
            IAssignmentsConsumerPtr consumer)
    {
        return new TUrlLocatorActor(balancerId, resolverId, std::move(cluster), std::move(consumer));
    }

} // namespace NSolomon::NFetcher
