#include <solomon/services/fetcher/lib/clients/access_service.h>
#include <solomon/services/fetcher/lib/config_updater/config_updater.h>
#include <solomon/services/fetcher/lib/multishard_pulling/auth_gatekeeper.h>
#include <solomon/services/fetcher/lib/sink/sink.h>

#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/testing/gtest/gtest.h>

#include <utility>

using namespace NActors;
using namespace NSolomon::NFetcher;
using namespace NSolomon;

using yandex::solomon::common::UrlStatusType;

const std::unordered_map<TString, TVector<TString>> PROVIDER_TO_SERVICE_ACCOUNTS = {
    { "provider1", { "sa1_1", "sa1_2" } },
    { "provider2", { "sa2_1", "sa2_2", "sa2_22" } },
};

// which folders does a SA have a permission to write to
const std::unordered_map<TString, TVector<TString>> SERVICE_ACCOUNT_TO_FOLDER = {
    { "sa1_1", { "folder1_1" } },
    { "sa1_2", { "folder1_2" } },

    {"sa2_1", { "folder2_1" }},
    {"sa2_2", { "folder2_2", "folder2_22" }},
};

const std::unordered_map<TString, TString> TOKEN_TO_SERVICE_ACCOUNT = {
    { "token1_1", "sa1_1" },
    { "token1_2", "sa1_2" },

    { "token2_1", "sa2_1" },
    { "token2_2", "sa2_2" },
    { "token2_22", "sa2_2" },
};

bool ContainsFolder(const TVector<TString>& folders, const TString& folder) {
    for (auto& f: folders) {
        if (f == folder) {
            return true;
        }
    }

    return false;
}

class TASGrpcClient: public IAccessServiceClient {
private:
    TAsyncAuthenticateResponse Authenticate(TString token) noexcept override {
        ++AuthenticationCallsCnt;

        auto it = TOKEN_TO_SERVICE_ACCOUNT.find(token);
        if (TStringBuf{token}.starts_with("invalid_token") || it == TOKEN_TO_SERVICE_ACCOUNT.end()) {
            return NThreading::MakeFuture<TAsyncAuthenticateResponse::value_type>(
                TAsyncAuthenticateResponse::value_type::FromError(EAccessServiceAuthErrorType::FailedAuth, "err"));
        }

        TIamAccount acc;
        acc.Type = NSolomon::EIamAccountType::Service;
        acc.Id = it->second;

        return NThreading::MakeFuture<TAsyncAuthenticateResponse::value_type>(
            TAsyncAuthenticateResponse::value_type::FromValue(std::move(acc)));
    }

    TAsyncAuthorizeResponse Authorize(const yandex::cloud::priv::servicecontrol::v1::AuthorizeRequest& req) noexcept override {
        ++AuthorizationCallsCnt;

        const auto& rp = req.resource_path(0);
        auto it = SERVICE_ACCOUNT_TO_FOLDER.find(req.subject().service_account().id());

        if (TStringBuf{rp.id()}.starts_with("invalid_folder") ||
            it == SERVICE_ACCOUNT_TO_FOLDER.end() ||
            !ContainsFolder(it->second, rp.id()))
        {
            return NThreading::MakeFuture<TAsyncAuthorizeResponse::value_type>(NGrpc::TGrpcStatus{"no write permission", 1, false});
        }

        yandex::cloud::priv::servicecontrol::v1::AuthorizeResponse resp;
        return NThreading::MakeFuture<TAsyncAuthorizeResponse::value_type>(std::move(resp));
    }

    void Stop(bool) override {
    }

public:
    size_t AuthenticationCallsCnt{0};
    size_t AuthorizationCallsCnt{0};
};

class TAuthGatekeeperTest: public ::testing::Test {
public:
    void SetUp() override {
        ActorRuntime = TTestActorRuntime::CreateInited(1, false, false);
        Registry = std::make_shared<NMonitoring::TMetricRegistry>();
        EdgeId = ActorRuntime->AllocateEdgeActor();
    }

    void TearDown() override {
        ActorRuntime.Reset();
    }

    TActorId CreateAndRegisterAuthGatekeeper(IAccessServiceClientPtr asClient, bool dontTakeAuthorizationIntoAccount = false) {
        TAccessServiceConfig config;
        config.SetRetriesCnt(0);
        config.SetMaxInflight(190);

        auto* d = config.MutableMinRetryDelay();
        d->SetValue(3);
        d->SetUnit(yandex::solomon::config::SECONDS);

        d = config.MutableMaxRetryDelay();
        d->SetValue(10);
        d->SetUnit(yandex::solomon::config::SECONDS);

        d = config.MutableTimeout();
        d->SetValue(10);
        d->SetUnit(yandex::solomon::config::SECONDS);

        TActorId asId = ActorRuntime->Register(CreateAccessServiceActor(
            std::move(asClient),
            TActorId{},
            std::move(config),
            std::make_shared<NMonitoring::TMetricRegistry>()).release());

        auto authGatekeeper = ActorRuntime->Register(CreateIamAuthGatekeeper(
            asId,
            {}, // configUpdaterId
            EdgeId,
            UnusedRecordsTtl,
            CacheDuration,
            GcInterval,
            dontTakeAuthorizationIntoAccount,
            Registry).release());
        ActorRuntime->WaitForBootstrap();

        TVector<TProviderConfigPtr> added{};
        TVector<TProviderConfigPtr> changed{};
        TVector<TString> removed{};

        for (const auto& [provider, saIds]: PROVIDER_TO_SERVICE_ACCOUNTS) {
            auto providerConfig = MakeAtomicShared<NDb::NModel::TProviderConfig>();
            providerConfig->Id = provider;
            providerConfig->IamServiceAccountIds = saIds;

            added.emplace_back(providerConfig);
        }

        auto providerEv = std::make_unique<TEvProvidersChanged>(added, changed, removed);
        ActorRuntime->Send(authGatekeeper, THolder(providerEv.release()));

        return authGatekeeper;
    }

    THolder<NSolomon::TTestActorRuntime> ActorRuntime;
    std::shared_ptr<NMonitoring::TMetricRegistry> Registry;
    TActorId EdgeId;

    TDuration UnusedRecordsTtl{TDuration::Minutes(5)};
    TDuration CacheDuration{TDuration::Minutes(5)};
    TDuration GcInterval{TDuration::Minutes(1)};
};

TEST_F(TAuthGatekeeperTest, HappyPath) {
    auto asClient = MakeIntrusive<TASGrpcClient>();
    auto authGatekeeper = CreateAndRegisterAuthGatekeeper(asClient);

    TShardData data;
    data.ProviderAccountId = "provider1";
    data.ClusterName = "folder1_2";

    auto ev = std::make_unique<TAuthGatekeeperEvents::TExamine>(data, "provider1", "token1_2", "url");
    ActorRuntime->Send(new IEventHandle{authGatekeeper, EdgeId, ev.release()});

    auto writeEv = ActorRuntime->GrabEdgeEvent<TEvSinkWrite>(EdgeId);
    ASSERT_TRUE(writeEv);

    ASSERT_EQ(writeEv->Get()->ShardData.ProviderAccountId, data.ProviderAccountId);
}

TEST_F(TAuthGatekeeperTest, ErrorsAreSentBack) {
    auto asClient = MakeIntrusive<TASGrpcClient>();
    auto authGatekeeper = CreateAndRegisterAuthGatekeeper(asClient);

    // invalid token
    {
        TShardData data;
        data.ProviderAccountId = "provider1";
        data.ClusterName = "folder1_2";

        auto ev = std::make_unique<TAuthGatekeeperEvents::TExamine>(data, data.ProviderAccountId, "invalid_token", "url");
        ActorRuntime->Send(new IEventHandle{authGatekeeper, EdgeId, ev.release()});

        auto errEv = ActorRuntime->GrabEdgeEvent<TEvMetricDataWritten>();
        ASSERT_TRUE(errEv);

        ASSERT_EQ(errEv->MetricsParsed, 0ul);
        ASSERT_EQ(errEv->Status, UrlStatusType::AUTH_ERROR);
    }

    // SA from a different provider
    {
        TShardData data;
        data.ProviderAccountId = "provider1";
        data.ClusterName = "folder1_2";

        auto ev = std::make_unique<TAuthGatekeeperEvents::TExamine>(data, data.ProviderAccountId, "token2_1", "url");
        ActorRuntime->Send(new IEventHandle{authGatekeeper, EdgeId, ev.release()});

        auto errEv = ActorRuntime->GrabEdgeEvent<TEvMetricDataWritten>();
        ASSERT_TRUE(errEv);

        ASSERT_EQ(errEv->MetricsParsed, 0ul);
        ASSERT_EQ(errEv->Status, UrlStatusType::AUTH_ERROR);
        ASSERT_THAT(errEv->Error, testing::HasSubstr("failed to match a provider id"));
    }

    // unknown provider
    {
        TShardData data;
        data.ProviderAccountId = "invalid_provider";
        data.ClusterName = "folder1_2";

        auto ev = std::make_unique<TAuthGatekeeperEvents::TExamine>(data, data.ProviderAccountId, "token2_2", "url");
        ActorRuntime->Send(new IEventHandle{authGatekeeper, EdgeId, ev.release()});

        auto errEv = ActorRuntime->GrabEdgeEvent<TEvMetricDataWritten>();
        ASSERT_TRUE(errEv);

        ASSERT_EQ(errEv->MetricsParsed, 0ul);
        ASSERT_EQ(errEv->Status, UrlStatusType::AUTH_ERROR);
        ASSERT_THAT(errEv->Error, testing::HasSubstr("no provider with id"));
    }

    // no write permission
    {
        TShardData data;
        data.ProviderAccountId = "provider1";
        data.ClusterName = "folder1_2";

        auto ev = std::make_unique<TAuthGatekeeperEvents::TExamine>(data, data.ProviderAccountId, "token1_1", "url");
        ActorRuntime->Send(new IEventHandle{authGatekeeper, EdgeId, ev.release()});

        auto errEv = ActorRuntime->GrabEdgeEvent<TEvMetricDataWritten>();
        ASSERT_TRUE(errEv);

        ASSERT_EQ(errEv->MetricsParsed, 0ul);
        ASSERT_EQ(errEv->Status, UrlStatusType::AUTH_ERROR);
        ASSERT_THAT(errEv->Error, testing::HasSubstr("no write permission"));
    }
}

TEST_F(TAuthGatekeeperTest, SkipAuthorization) {
    auto asClient = MakeIntrusive<TASGrpcClient>();
    auto authGatekeeper = CreateAndRegisterAuthGatekeeper(asClient, /*dontTakeAuthorizationIntoAccount = */true);

    TShardData data;
    data.ProviderAccountId = "provider1";
    data.ClusterName = "folder1_2"; // folder for sa1_2, but we will write to it from sa1_1

    auto ev = std::make_unique<TAuthGatekeeperEvents::TExamine>(data, data.ProviderAccountId, "token1_1", "url");
    ActorRuntime->Send(new IEventHandle{authGatekeeper, EdgeId, ev.release()});

    auto succEv = ActorRuntime->GrabEdgeEvent<TEvSinkWrite>(TDuration::Seconds(1));
    ASSERT_TRUE(succEv); // we still get the OK response, even though authorization failed
}

TEST_F(TAuthGatekeeperTest, Cache) {
    auto asClient = MakeIntrusive<TASGrpcClient>();
    auto authGatekeeper = CreateAndRegisterAuthGatekeeper(asClient);

    auto sendReq = [&](TString provider, TString folder, const TString& token, const TString& url) {
        TShardData data;
        data.ProviderAccountId = std::move(provider);
        data.ClusterName = std::move(folder);

        auto ev = std::make_unique<TAuthGatekeeperEvents::TExamine>(data, data.ProviderAccountId, token, url);
        ActorRuntime->Send(new IEventHandle{authGatekeeper, EdgeId, ev.release()});

        ActorRuntime->GrabEdgeEvent<TEvSinkWrite>(TDuration::Seconds(1));
    };

    sendReq("provider2", "folder2_2", "token2_2", "url");
    ASSERT_EQ(asClient->AuthenticationCallsCnt, 1ul);
    ASSERT_EQ(asClient->AuthorizationCallsCnt, 1ul);

    sendReq("provider2", "folder2_2", "token2_2", "url");
    ASSERT_EQ(asClient->AuthenticationCallsCnt, 1ul); // req cnt hasn't been increased -- cache hit
    ASSERT_EQ(asClient->AuthorizationCallsCnt, 1ul);

    ActorRuntime->AdvanceCurrentTime(CacheDuration + TDuration::Seconds(5));

    sendReq("provider2", "folder2_2", "token2_2", "url");
    ASSERT_EQ(asClient->AuthenticationCallsCnt, 2ul); // cache is stale -- cache miss
    ASSERT_EQ(asClient->AuthorizationCallsCnt, 2ul);

    // different url, but data is the same
    sendReq("provider2", "folder2_2", "token2_2", "url2");
    ASSERT_EQ(asClient->AuthenticationCallsCnt, 2ul); // cache hit
    ASSERT_EQ(asClient->AuthorizationCallsCnt, 2ul); // cache miss

    // different token, but sa is the same
    sendReq("provider2", "folder2_2", "token2_22", "url");
    ASSERT_EQ(asClient->AuthenticationCallsCnt, 3ul); // cache miss
    ASSERT_EQ(asClient->AuthorizationCallsCnt, 2ul); // cache hit, 'cause a sa is the same

    // same token, but different folder
    sendReq("provider2", "folder2_22", "token2_2", "url");
    ASSERT_EQ(asClient->AuthenticationCallsCnt, 3ul); // cache hit
    ASSERT_EQ(asClient->AuthorizationCallsCnt, 3ul); // cache miss
}
