#include "queue.h"

#include <solomon/services/fetcher/lib/sink/sink.h>

namespace NSolomon::NFetcher {
namespace {
    struct TFakeCounters: TShardQueue::ICounters {
        void AddSize(i64) override {
        }

        void IncSize() override {
        }

        void DecSize() override {
        }
    };

    struct TFakeMemoryLimiter: IQueueMemoryLimiter {
        bool OnWrite(ui64) override {
            return true;
        }

        void OnFree(ui64) override {
        }
    };
} // namespace

    TQueueEntry::TQueueEntry(TEvSinkWrite&& ev, NActors::TActorId sender)
        : Sender{sender}
        , Data{std::move(ev.ShardData.Data)}
        , Host{std::move(ev.ShardData.Host)}
        , Url{std::move(ev.ShardData.Url)}
        , Interval{std::move(ev.ShardData.Interval)}
        , Instant{std::move(ev.ShardData.Instant)}
        , Format{std::move(ev.ShardData.Format)}
        , Labels{std::move(ev.ShardData.Labels)}
        , PrevData{std::move(ev.ShardData.PrevData)}
        , PrevInstant{std::move(ev.ShardData.PrevInstant)}
        , SourceId{std::move(ev.ShardData.SourceId)}
    {
    }

    TShardQueue::TShardQueue(size_t perUrlLimit, size_t capacity, THolder<ICounters> counters, const IQueueMemoryLimiterPtr& limiter)
        : PerUrlLimit_{perUrlLimit}
        , Capacity_{capacity}
        , Counters_{counters ? std::move(counters) : CreateFakeCounters()}
        , MemoryLimiter_{limiter ? limiter : CreateFakeMemoryLimiter()}
    {
    }

    TShardQueue::~TShardQueue() {
        Clear();
    }

    void TShardQueue::Clear() {
        Counters_->AddSize(-static_cast<i64>(Queue_.size()));
        MemoryLimiter_->OnFree(MemorySize());

        Queue_.clear();
        UrlQueueSize_.clear();
        MemorySize_ = 0;
    }

    bool TShardQueue::IsEmpty() const {
        return Queue_.empty();
    }

    bool TShardQueue::IsFull() const {
        return Capacity_ != 0 && Queue_.size() >= Capacity_;
    }

    TQueueEntry TShardQueue::Pop() {
        auto val = std::move(Queue_.front());
        auto* size = UrlQueueSize_.FindPtr(val.Url);
        Y_VERIFY_DEBUG(size != nullptr);
        if (size) {
            --(*size);
        }

        Counters_->DecSize();

        Queue_.pop_front();
        MemorySize_ -= val.Size();
        MemoryLimiter_->OnFree(val.Size());

        return val;
    }

    ui64 TShardQueue::MemorySize() const {
        return MemorySize_;
    }

    ui64 TShardQueue::EntrySize() const {
        return Queue_.size();
    }

    THolder<TShardQueue::ICounters> CreateFakeCounters() {
        return MakeHolder<TFakeCounters>();
    }

    IQueueMemoryLimiterPtr CreateFakeMemoryLimiter() {
        return new TFakeMemoryLimiter;
    }
} // namespace NSolomon::NFetcher
