#pragma once

#include <solomon/services/fetcher/lib/source_id/factory.h>

#include <library/cpp/actors/core/actorid.h>
#include <library/cpp/monlib/encode/format.h>
#include <library/cpp/monlib/metrics/labels.h>

#include <util/datetime/base.h>
#include <util/generic/string.h>
#include <util/generic/deque.h>

namespace NSolomon::NFetcher {
    struct TQueueEntry {
        TQueueEntry() = default;
        TQueueEntry(struct TEvSinkWrite&& data, NActors::TActorId sender);

        TQueueEntry(const TQueueEntry&) = default;
        TQueueEntry(TQueueEntry&&) = default;
        TQueueEntry& operator=(const TQueueEntry&) = default;
        TQueueEntry& operator=(TQueueEntry&&) = default;

        size_t Size() const {
            return Data.size() + PrevData.size();
        }

        NActors::TActorId Sender;

        // inlined TEvSinkWrite
        TString Data;
        TString Host;
        TString Url;
        TDuration Interval;
        TInstant Instant;

        NMonitoring::EFormat Format;
        NMonitoring::TLabels Labels;

        TString PrevData;
        TInstant PrevInstant;
        TSourceId SourceId;
    };

    class IQueueMemoryLimiter: public TThrRefBase {
    public:
        virtual bool OnWrite(ui64 bytes) = 0;
        virtual void OnFree(ui64 bytes) = 0;
    };

    using IQueueMemoryLimiterPtr = TIntrusivePtr<IQueueMemoryLimiter>;

    // this queue is supposed to be used from an actor, so is NOT thread-safe
    class TShardQueue: public TMoveOnly {
    public:
        struct ICounters {
            virtual ~ICounters() = default;
            virtual void AddSize(i64) = 0;
            virtual void IncSize() = 0;
            virtual void DecSize() = 0;
        };

        // 0 means no limit
        TShardQueue(size_t perUrlLimit, size_t capacity, THolder<ICounters> size, const IQueueMemoryLimiterPtr& limiter = nullptr);
        ~TShardQueue();

        template <typename... TArgs>
        bool Push(TArgs&&... args) {
            if (IsFull()) {
                return false;
            }

            TQueueEntry entry{std::forward<TArgs>(args)...};

            auto [it, _] = UrlQueueSize_.emplace(entry.Url, 0);
            if (PerUrlLimit_ != 0 && it->second >= PerUrlLimit_) {
                return false;
            }

            if (!MemoryLimiter_->OnWrite(entry.Size())) {
                return false;
            }

            it->second++;

            Counters_->IncSize();
            MemorySize_ += entry.Size();

            Queue_.push_back(std::move(entry));
            return true;
        }

        TQueueEntry Pop();

        ui64 MemorySize() const;
        ui64 EntrySize() const;

        void Clear();

        bool IsEmpty() const;
        bool IsFull() const;

    private:
        TDeque<TQueueEntry> Queue_;
        size_t PerUrlLimit_;
        size_t Capacity_;
        THolder<ICounters> Counters_;
        THashMap<TString, size_t> UrlQueueSize_;
        ui64 MemorySize_{0};
        IQueueMemoryLimiterPtr MemoryLimiter_;
    };

    THolder<TShardQueue::ICounters> CreateFakeCounters();
    IQueueMemoryLimiterPtr CreateFakeMemoryLimiter();
} // namespace NSolomon::NFetcher
