#include <solomon/services/fetcher/lib/racktables/dc_matcher.h>
#include <solomon/libs/cpp/http/client/curl/client.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/ipv6_address/ipv6_address.h>
#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/resource/resource.h>
#include <library/cpp/threading/future/future.h>

#include <util/stream/file.h>

using namespace NSolomon;

void Dump(IDcMatcher* matcher) {
    matcher->Dump(Cout);
}

void Locate(const TString& addrStr, IDcMatcher* matcher) {
    bool ok{};
    auto addr = TIpv6Address::FromString(addrStr, ok);
    if (!ok) {
        Cerr << "cannot parse " << addrStr << " as an IP address";
        exit(1);
    } else if (addr.Type() == TIpv6Address::TIpType::Ipv4) {
        Cerr << "Only IPv6 addrs are supported" << Endl;
    }

    Cout << addr << " -> " << matcher->DcByAddress(addr) << Endl;
}

TString GetResponseData(IHttpClient::TResult result) {
    if (!result.Success()) {
        auto& err = result.Error();
        Cerr << "Request failed: " << static_cast<size_t>(err.Type()) << " " << result.Error().Message() << Endl;
        return "";
    }
    auto response = result.Extract();
    return response->ExtractData();
}

TString GetIpData() {
    const auto& url = "https://ro.racktables.yandex.net/export/networklist.php?report=net_places";
    auto h = Headers({{"User-Agent","Mozilla/5.0"}});

    NMonitoring::TMetricRegistry r;
    IHttpClientPtr client = CreateCurlClient({}, r);
    auto promise = NThreading::NewPromise<TString>();
    auto future = promise.GetFuture();
    client->Request(
            CreateRequest(EHttpMethod::Get, url, {}, std::move(h)),
            [promise] (auto result) mutable {
                promise.SetValue(GetResponseData(std::move(result)));
            }
    );
    return future.ExtractValueSync();
}

int main(int argc, char** argv) {
    using namespace NLastGetopt;

    bool dump{false};

    TString file;
    TString ipAddr;

    TOpts opts = NLastGetopt::TOpts::Default();
    opts.AddHelpOption('h');
    opts.AddLongOption('f', "file", "load tables from file")
        .RequiredArgument("FILE")
        .StoreResult(&file)
        .DefaultValue("");
    opts.AddLongOption("dump", "dump all DC networks")
        .StoreTrue(&dump)
        .NoArgument();
    opts.AddLongOption("locate")
        .StoreResult(&ipAddr)
        .RequiredArgument("IPADDR");
    opts.SetFreeArgsNum(0);

    TOptsParseResult r(&opts, argc, argv);

    TString data;
    if (file.empty()) {
        data = GetIpData();
    } else {
        data = TFileInput{file}.ReadAll();
    }

    const auto matcher = CreateIpv6Matcher(static_cast<TStringBuf>(data));
    if (ipAddr) {
        Locate(ipAddr, matcher.Get());
    }

    if (dump) {
        Dump(matcher.Get());
    }
}
