#include "dc_matcher.h"

#include <library/cpp/ipmath/range_set.h>

#include <library/cpp/ipv6_address/ipv6_address.h>
#include <library/cpp/resource/resource.h>

#include <util/string/split.h>
#include <util/charset/utf8.h>
#include <util/generic/hash.h>
#include <util/stream/file.h>

namespace NSolomon {
namespace {
    EDc GuessFromString(TStringBuf str) {
        static const THashMap<TString, EDc> DC_RUSSIAN_NAMES = {
            {"угрешка", EDc::UGR},
            {"фольга", EDc::FOL},
            {"ивантеевка", EDc::IVA},
            {"мытищи", EDc::MYT},
            {"эшбёрн", EDc::ASH},
            {"амстердам", EDc::AMS},
            {"лас-вегас", EDc::VEG},
            {"мянтсяля", EDc::MAN},
            {"владимир", EDc::VLA},
            {"сасово", EDc::SAS},
        };

        const auto norm = ToLowerUTF8(str);

        for (auto&& [name, val]: DC_RUSSIAN_NAMES) {
            if (norm.Contains(name)) {
                return val;
            }
        }

        return EDc::UNKNOWN;
    }

    class TDcMatcher: public IDcMatcher {
    public:
        void Parse(TStringBuf data) {
            for (auto&& tok: StringSplitter(data).Split('\n').SkipEmpty()) {
                auto&& row = tok.Token();

                // string looks like 5.45.192.0/22    [Угрешка]
                TStringBuf net, name;
                const auto ok = StringSplitter(row)
                        .SplitBySet(" \t")
                        .Limit(2)
                        .SkipEmpty()
                        .TryCollectInto(&net, &name);

                if (!ok) {
                    continue;
                }

                const auto dc = GuessFromString(name);

                if (dc == EDc::UNKNOWN) {
                    continue;
                }

                auto addr = TIpAddressRange::FromCidrString(TString{net});
                if (addr.Type() != TIpAddressRange::TIpType::Ipv6) {
                    continue;
                }

                DcNets_[dc].Add(addr);
            }
        }

        explicit TDcMatcher(TStringBuf data) {
            if (!data.empty()) {
                Parse(data);
            }
        }

        EDc DcByAddress(const TIpv6Address& addr) const override {
            for (auto&& [dc, rangeSet]: DcNets_) {
                if (rangeSet.Contains(addr)) {
                    return dc;
                }
            }

            return EDc::UNKNOWN;
        }

        void Dump(IOutputStream& os) const override {
            for (auto&& [dc, rangeSet]: DcNets_) {
                os << '[' << dc << "]\n";

                for (auto&& range: rangeSet) {
                    os << range.ToRangeString() << Endl;
                }
            }
        }

        size_t Size() const override {
            return DcNets_.size();
        }

    private:
        THashMap<EDc, TIpRangeSet> DcNets_;
    };

} // namespace

IDcMatcherPtr CreateIpv6Matcher(TStringBuf data) {
    return MakeHolder<TDcMatcher>(data);
}

IDcMatcherPtr CreateDefaultIpv6Matcher() {
    auto data = NResource::Find(TStringBuf("net_places.txt"));
    return MakeHolder<TDcMatcher>(data);
}

IDcMatcherPtr CreateIpv6Matcher(const TFsPath& file) {
    auto data = TFileInput{file}.ReadAll();
    return MakeHolder<TDcMatcher>(data);
}

} // namespace NSolomon
