#include "racktables_actor.h"
#include "racktables_metrics.h"
#include "dc_matcher.h"

#include <solomon/libs/cpp/backoff/jitter.h>
#include <solomon/libs/cpp/backoff/backoff.h>
#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/steady_timer/steady_timer.h>

#include <build/scripts/c_templates/svnversion.h>

#include <contrib/libs/openssl/include/openssl/sha.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>

#include <util/stream/file.h>

using namespace NActors;
using namespace NMonitoring;

namespace NSolomon::NFetcher {
namespace {

struct TResolveRequest {
    TActorId ReplyTo;
    TIpv6Address Address;
    ui32 Flags = 0;
    ui64 Cookie = 0;
};

TString ShaSum(const TString& data) {
    SHA256_CTX sha256;
    SHA256_Init(&sha256);
    SHA256_Update(&sha256, data.data(), data.size());
    TString hash(SHA256_DIGEST_LENGTH, '\0');
    SHA256_Final(reinterpret_cast<unsigned char*>(hash.begin()), &sha256);
    return hash;
}

class TRackTablesActor: public TActorBootstrapped<TRackTablesActor>, private TPrivateEvents {
    enum {
        DcResponse = SpaceBegin,
        Error,
        End,
    };

    struct TResponse: public NActors::TEventLocal<TResponse, DcResponse> {
        const TString Data;
        const HttpCodes Code;
        const TDuration DownloadTime;

        explicit TResponse(TString data, HttpCodes code, TDuration downloadTime) noexcept
            : Data{std::move(data)}
            , Code{code}
            , DownloadTime{downloadTime}
        {
        }
    };

    struct TError: public NActors::TEventLocal<TError, Error> {
        const TString Message;
        const HttpCodes Code;
        const TDuration DownloadTime;

        explicit TError(TString Message, HttpCodes code, TDuration downloadTime) noexcept
            : Message{std::move(Message)}
            , Code{code}
            , DownloadTime{downloadTime}
        {
        }
    };

public:
    explicit TRackTablesActor(const TRackTablesActorConf& conf)
        : HttpClient_{conf.HttpClient}
        , ConnectTimeout_{conf.ConnectTimeout}
        , ReadTimeout_{conf.ReadTimeout}
        , Retries_{conf.Retries}
        , Url_{conf.Url}
        , FileCache_{conf.FileCache}
        , UserAgent_{TStringBuilder() << "Solomon-Fetcher/" << GetProgramSvnRevision()}
        , Backoff_{conf.RefreshInterval, 5 * conf.RefreshInterval}
        , Metrics_{conf.Registry}
    {
        if (FileCache_) {
            try {
                TString data = TFileInput{FileCache_}.ReadAll();
                try {
                    TSteadyTimer timer;
                    Matcher_ = CreateIpv6Matcher(static_cast<TStringBuf>(data));
                    Metrics_.WriteParseStatus(true);
                    Metrics_.WriteParseTime(timer.Step());
                    Metrics_.WriteRows(Matcher_->Size());
                } catch (...) {
                    Metrics_.WriteParseStatus(false);
                }
                DataHash_ = ShaSum(data);
            } catch (...) {
                Cerr << "cannot racktables cache from " << FileCache_ << ": " << CurrentExceptionMessage();
            }
        }
    }

    void Bootstrap() {
        LastUpdate_ = TActivationContext::Now();
        Become(&TThis::StateFunc);
        Schedule(Backoff_(), new TEvents::TEvWakeup);
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            sFunc(TEvents::TEvWakeup, OnWakeUp)
            hFunc(TEvents::TEvPoison, OnPoison)
            hFunc(TResponse, OnResponse)
            hFunc(TRackTablesEvents::TDcResolve, OnRequest)
            hFunc(TError, OnError)
        }
    }

    void OnWakeUp() {
        Metrics_.WriteAge(TActivationContext::Now() - LastUpdate_);
        Refresh();
    }

    void OnPoison(const TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new TEvents::TEvPoisonTaken, ev->Flags, ev->Cookie);
        PassAway();
    }

    void Refresh() {
        Schedule(Backoff_(), new TEvents::TEvWakeup);
        auto* as = TActivationContext::ActorSystem();
        auto headers = Headers();

        headers->Add(TStringBuf("User-Agent"), UserAgent_);
        TRequestOpts opts{
            .ConnectTimeout = ConnectTimeout_,
            .ReadTimeout = ReadTimeout_,
            .Retries = Retries_,
        };

        TSteadyTimer timer;
        HttpClient_->Request(
                CreateRequest(EHttpMethod::Get, Url_, {}, std::move(headers)),
                [as, selfId = SelfId(), timer](auto result) mutable {
                    try {
                        if (!result.Success()) {
                            as->Send(selfId,
                                new TError{TStringBuilder() << "Server error " << result.Error().Message(), HTTP_CODE_MAX, timer.Step()}
                            );
                            return;
                        }
                        auto response = result.Extract();
                        if (response->Code() != HTTP_OK) {
                            as->Send(selfId,
                                    new TError{TStringBuilder() << "Server returned with " << response->Code() << " code", response->Code(), timer.Step()}
                            );
                        } else {
                            as->Send(selfId, new TResponse{response->ExtractData(), HTTP_OK, timer.Step()});
                        }
                    } catch (...) {
                        as->Send(selfId, new TError{CurrentExceptionMessage(), HTTP_CODE_MAX, timer.Step()});
                    }
                },
                opts);
    }

    void OnRequest(const NSolomon::NFetcher::TRackTablesEvents::TDcResolve::TPtr& ev) {
        if (!Matcher_) {
            PostponedRequests_.emplace_back(TResolveRequest{ev->Sender, ev->Get()->Address, ev->Flags, ev->Cookie});
            return;
        }
        auto dc = Matcher_->DcByAddress(ev->Get()->Address);
        Send(ev->Sender, new TRackTablesEvents::TDcResult(dc), ev->Flags, ev->Cookie);
        Metrics_.WriteResolveDc(dc);
    }

    void OnResponse(const TResponse::TPtr& ev) {
        Backoff_.Reset();
        Metrics_.WriteCode(ev->Get()->Code);
        Metrics_.WriteDownloadTime(ev->Get()->DownloadTime);
        try {
            TString newDataHash = ShaSum(ev->Get()->Data);
            if (DataHash_ != newDataHash) {
                DataHash_ = newDataHash;
                TSteadyTimer timer;
                Matcher_ = CreateIpv6Matcher(static_cast<TStringBuf>(ev->Get()->Data));
                Metrics_.WriteParseTime(timer.Step());
                MON_INFO(RackTables, "Racktables data has been successfully updated");
                Metrics_.WriteParseStatus(true);
                Metrics_.WriteRows(Matcher_->Size());
                if (FileCache_) {
                    try{
                        TFileOutput out{FileCache_};
                        out.Write(ev->Get()->Data);
                        out.Finish();
                        Metrics_.WriteCacheStatus(true);
                    } catch(...) {
                        MON_WARN(RackTables, "Error while writing racktables cache: " << CurrentExceptionMessage());
                        Metrics_.WriteCacheStatus(false);
                    }
                }

                if (!PostponedRequests_.empty()) {
                    for (const auto& req: PostponedRequests_) {
                        auto dc = Matcher_->DcByAddress(req.Address);
                        Send(req.ReplyTo, new TRackTablesEvents::TDcResult{dc}, req.Flags, req.Cookie);
                        Metrics_.WriteResolveDc(dc);
                    }
                    PostponedRequests_.clear();
                }
            } else {
                MON_INFO(RackTables, "Racktables data hasn't changed since last update");
            }
            LastUpdate_ = TActivationContext::Now();
        } catch (...) {
            MON_WARN(RackTables, "Parser error : " << CurrentExceptionMessage());
            Metrics_.WriteParseStatus(false);
        }
    }

    void OnError(TError::TPtr ev) {
        MON_WARN(RackTables, "cannot get rack tables data from " << Url_ << ", message: " << ev->Get()->Message);
        Metrics_.WriteCode(ev->Get()->Code);
        Metrics_.WriteDownloadTime(ev->Get()->DownloadTime);
    }

private:
    IHttpClientPtr HttpClient_;
    IDcMatcherPtr Matcher_;
    TInstant LastUpdate_;
    TDuration ConnectTimeout_;
    TDuration ReadTimeout_;
    ui8 Retries_;
    TString Url_;
    TFsPath FileCache_;
    std::vector<TResolveRequest> PostponedRequests_;
    const TString UserAgent_;
    TLinearBackoff<THalfJitter> Backoff_;
    TString DataHash_;
    TRackTablesMetrics Metrics_;
};

class TRacktablesActorStub: public TActor<TRacktablesActorStub>, private TPrivateEvents {
public:
    explicit TRacktablesActorStub(TMetricRegistry& registry)
        : TActor(&TThis::StateFunc)
        , Metrics_{registry}
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TEvents::TEvPoison, OnPoison)
            hFunc(TRackTablesEvents::TDcResolve, OnRequest)
        }
    }

    void OnPoison(const TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new TEvents::TEvPoisonTaken, ev->Flags, ev->Cookie);
        PassAway();
    }

    void OnRequest(const NSolomon::NFetcher::TRackTablesEvents::TDcResolve::TPtr& ev) {
        Send(ev->Sender, new TRackTablesEvents::TDcResult(EDc::UNKNOWN), ev->Flags, ev->Cookie);
        Metrics_.WriteResolveDc(EDc::UNKNOWN);
    }

private:
    TRackTablesMetrics Metrics_;
};
} // namespace

std::unique_ptr<IActor> CreateRackTablesActor(const TRackTablesActorConf& conf) {
    return std::make_unique<TRackTablesActor>(conf);
}

std::unique_ptr<IActor> CreateRacktablesActorStub(TMetricRegistry& registry) {
    return std::make_unique<TRacktablesActorStub>(registry);
}

} // namespace solomon
